import React, { FC, useRef, useState, useContext } from 'react';
import cx from 'classnames';
import Popup from '@crm/components/dist/lego2/Popup';
import { HeaderCellProps } from './HeaderCellWithControls.types';
import { context } from '../../services/TableController';
import { UserFilter } from './UserFilter';
import { getNextSortOrder } from './HeaderCellWithControls.utils';
import { ControlIcon } from './ControlIcon';
import themes from './themes';
import { themeContext } from '../../themes';
import { config } from 'services/Config';

export const HeaderCellWithControls: FC<HeaderCellProps> = ({ column, className }) => {
  const theme = useContext(themeContext);
  const controller = useContext(context);
  const css = themes[theme] || themes.default;
  const cellRef = useRef<HTMLDivElement>(null);
  const [filterVisibility, setFilterVisibility] = useState(false);

  const filter = controller.getUserFilterById(column.id);
  const sort = controller.getSortById(column.id);

  const isFiltered = Boolean(filter?.data);
  const isSortable = Boolean(column.sortable);
  const showIcon = isSortable || filter;

  const cellCn = cx(css.HeaderCellWithControls, className, {
    [css.HeaderCellWithControls_sticky]: column.isPinned,
    [css.HeaderCellWithControls__view_clickable]: isSortable || filter,
  });

  const handleClick = () => {
    if (filter) {
      setFilterVisibility(!filterVisibility);
      return;
    }
    if (isSortable) {
      const nextSortOrder = getNextSortOrder(sort);
      controller.addSort({ id: column.id, order: nextSortOrder }).fetch();
    }
  };

  const handleOutsideClick = () => {
    setFilterVisibility(false);
  };

  return (
    <>
      <div ref={cellRef} className={cellCn} key={column.id} onClick={handleClick}>
        <span>{column.title}</span>
        {showIcon && <ControlIcon sort={sort} isFiltered={isFiltered} />}
      </div>
      {filterVisibility && (
        <Popup
          direction={['bottom-start', 'bottom-end', 'top-start', 'top-end']}
          anchor={cellRef}
          target="anchor"
          visible={filterVisibility}
          onOutsideClick={handleOutsideClick}
          continuePropagation={config.value.features.newDesignLift}
          className={css.HeaderCellWithControls__popup}
        >
          {filter && (
            <UserFilter
              closePopup={handleOutsideClick}
              controller={controller}
              column={column}
              filter={filter}
              sort={sort}
            />
          )}
        </Popup>
      )}
    </>
  );
};
