import React, { FC, useState } from 'react';
import Button from '@crm/components/dist/lego2/Button';
import { UserFilterAny, UserFilterSearchType } from 'types/api/form/UserFilter';
import { SortObject } from 'types/api/form/Form';
import isEqual from 'lodash/isEqual';
import { getFilterComponents } from './adapters';
import css from './UserFilter.module.css';
import { UserFiltersProps } from './UserFilter.types';
import { UNKNOWN_FILTER_TYPE_MESSAGE } from './UserFilter.constants';
import { SortPanel } from './SortPanel';
import { SearchType } from './SearchType';

export const UserFilter: FC<UserFiltersProps> = ({
  closePopup,
  sort,
  column,
  filter,
  controller,
}) => {
  const FilterComponent = getFilterComponents()[filter.type];
  const [filterData, setFilterData] = useState<UserFilterAny | undefined>(filter);
  const [sortData, setSortData] = useState<SortObject | undefined>(sort);
  const [searchType, setSearchType] = useState<UserFilterSearchType>(filter.searchType!);

  if (!FilterComponent) {
    return <span>{UNKNOWN_FILTER_TYPE_MESSAGE}</span>;
  }
  const isNotChanged =
    isEqual(sort, sortData) &&
    isEqual(filter, filterData) &&
    isEqual(searchType, filter.searchType);

  const handleClick = () => {
    if (filterData) {
      const newFilter = { ...filterData };
      if (filter.enableSearchExtension) {
        newFilter.enableSearchExtension = true;
        newFilter.searchType = searchType;
      }

      controller.addUserFilter(column.id, newFilter).resetCurrentPage();
    } else {
      controller.removeUserFilter(column.id).resetCurrentPage();
    }
    if (sortData) {
      controller.addSort(sortData);
    } else {
      controller.removeSort(column.id);
    }
    controller.fetch();
    closePopup();
  };

  const handleReset = () => {
    controller
      .removeUserFilter(column.id)
      .removeSort(column.id)
      .fetch();
    closePopup();
  };

  if (FilterComponent) {
    const enableSearchExt = Boolean(filter.enableSearchExtension);
    const displayFilterComponent = !enableSearchExt || ['Equal', 'NotEqual'].includes(searchType);

    return (
      <div className={css.UserFilter} data-testid={'TableUserFilter'}>
        <div className={css.UserFilter__inputWrapper}>
          {enableSearchExt && (
            <SearchType
              className={css.UserFilter__searchType}
              value={searchType}
              onChange={setSearchType}
            />
          )}
          {displayFilterComponent && (
            <FilterComponent column={column} filter={filterData} changeFilter={setFilterData} />
          )}
        </div>
        {column.sortable && <SortPanel column={column} changeSort={setSortData} sort={sortData} />}
        <div className={css.UserFilter__buttonsPanel}>
          <Button data-testid="resetButton" view="default" onClick={handleReset} size="m">
            Сбросить
          </Button>
          <Button
            data-testid="applyButton"
            view="action"
            onClick={handleClick}
            disabled={isNotChanged}
            size="m"
          >
            Применить
          </Button>
        </div>
      </div>
    );
  }
  return null;
};
