import React, { ReactNodeArray } from 'react';
import { render, cleanup, waitFor, screen, fireEvent } from '@testing-library/react';
import { Tree } from './Tree';
import { TreeDictStub } from './stubs/TreeDictStub';

jest.mock('react-virtualized-auto-sizer', () => {
  return jest.fn().mockImplementation(({ children }) => {
    return <div>{children(500, 500)}</div>;
  });
});

const mockToggle = jest.fn();

jest.mock('react-vtree', () => {
  return {
    VariableSizeTree: jest.fn().mockImplementation((props) => {
      const NodeComponent = props.children;
      const collection: ReactNodeArray = [];
      if (collection.length === 0) {
        const it = props.treeWalker(true);
        for (const item of it) {
          collection.push(<NodeComponent toggle={mockToggle} key={item.id} data={item} />);
        }
      }
      return <div>{collection}</div>;
    }),
  };
});

const onNodeClick = jest.fn();

const TreeProps = {
  tree: TreeDictStub.items,
  onNodeClick: onNodeClick,
  isForceOpen: false,
  selectedIds: new Set<string>(),
};

describe('Tree', () => {
  afterEach(() => {
    cleanup();
    jest.clearAllMocks();
  });

  it('renders root nodes', async () => {
    render(<Tree {...TreeProps} />);
    await waitFor(() => {
      expect(screen.getByText('item1')).toBeInTheDocument();
    });
    await waitFor(() => {
      expect(screen.queryByText('item3')).not.toBeInTheDocument();
    });
  });
  describe('when click on node', () => {
    it('calls onNodeClick', async () => {
      render(<Tree {...TreeProps} />);
      await waitFor(() => {
        fireEvent.click(screen.getByText('item1'));
      });
      await waitFor(() => {
        expect(onNodeClick).toHaveBeenCalledTimes(1);
      });
    });
  });
  describe('when click on expand arrow', () => {
    it('calls toggle on node', async () => {
      render(<Tree {...TreeProps} />);
      await waitFor(() => {
        fireEvent.click(screen.getByTestId('TreeNode_expandArrow_1'));
      });
      await waitFor(() => {
        expect(mockToggle).toHaveBeenCalledTimes(1);
      });
    });
  });
  describe('when isForceOpen', () => {
    it('renders all nodes', async () => {
      render(<Tree {...TreeProps} isForceOpen />);
      await waitFor(() => {
        expect(screen.getByText('item6')).toBeInTheDocument();
      });
      await waitFor(() => {
        expect(screen.getByText('item3')).toBeInTheDocument();
      });
    });
  });
});
