import React, { FC, useState, useEffect } from 'react';
import { Overlay } from 'components/Overlay';
import filterTree from 'utils/filter-tree';
import { Tree } from './Tree';
import { SelectedNodeList } from './SelectedNodeList';
import {
  TreeSuggestMultipleProps,
  TreeSuggestMultipleFilter,
  TreeNode,
} from './TreeSuggestMultiple.types';
import css from './TreeSuggestMultiple.module.css';
import { fetchTreeDict } from './utils/fetchTreeDict';
import { SearchInput } from './SearchInput';

export const TreeSuggestMultiple: FC<TreeSuggestMultipleProps> = ({ changeFilter, filter }) => {
  const [tree, setTree] = useState<TreeNode[] | undefined>();
  const [isLoading, setLoading] = useState(false);
  const [query, setQuery] = useState('');

  useEffect(() => {
    const request = fetchTreeDict(filter?.provider || '', setTree, setLoading);
    return () => {
      request.cancel();
    };
  }, []);

  const selectedIds = new Set(filter?.data?.value?.map((item) => String(item.id)) || []);

  const removeNode = (node: Omit<TreeNode, 'items'>) => {
    const newFilter: TreeSuggestMultipleFilter = { type: filter.type, provider: filter.provider };
    const newFilterValue = filter?.data?.value.filter((item) => item.id !== node.id);
    if (newFilterValue && newFilterValue.length > 0) {
      newFilter.data = { value: newFilterValue };
    }
    changeFilter(newFilter);
  };

  const handleChangeFilter = (node: TreeNode) => {
    const { items, ...selectedNode } = node;
    if (selectedIds.has(selectedNode.id)) {
      removeNode(selectedNode);
      return;
    }
    let data = filter.data;
    data = data ? { value: [...data.value, selectedNode] } : { value: [selectedNode] };
    changeFilter({ type: filter.type, provider: filter.provider, data });
  };

  return (
    <div className={css.TreeSuggestMultiple}>
      <SearchInput setSearchQuery={setQuery} />
      <div className={css.TreeSuggestMultiple__treeWrapper}>
        <Overlay display={isLoading} hasSpinner />
        {tree && (
          <Tree
            tree={filterTree(tree, query)}
            isForceOpen={Boolean(query)}
            onNodeClick={handleChangeFilter}
            selectedIds={selectedIds}
          />
        )}
      </div>
      {filter?.data && <SelectedNodeList onRemove={removeNode} filter={filter} />}
    </div>
  );
};
