import React, { FC, useMemo, useState } from 'react';
import { observer } from 'mobx-react-lite';
import { useBunker } from 'utils/hooks/useBunker';
import { PaginationPage } from 'types/api/form/Form';
import Spin from '@crm/components/dist/lego2/Spin';
import { config } from 'services/Config';
import { TableEditable } from '../TableEditable';
import { ThemeProvider } from '../themes';
import css from './TableWithFilters.module.css';
import { TableWithFiltersProps } from './TableWithFilters.types';
import { BunkerTableSettings } from '../Table.types';
import { TableController, Provider } from '../services/TableController';
import { withPagination } from '../hoc';
import { TemplateFilters } from './TemplateFilters';
import { getFiltersUrl } from './TableWithFilters.utils';
import { getBunkerNodeKeys } from '../utils/getBunkerNodeKey';
const Table = withPagination(TableEditable);

export const TableWithFilters: FC<TableWithFiltersProps> = observer(
  ({ url, title, showExtendsFilters = true }) => {
    const nodeKeys = useMemo(() => {
      return getBunkerNodeKeys(url);
    }, [url]);

    const bunkerData = useBunker<BunkerTableSettings>(nodeKeys);
    const [controller] = useState<TableController>(() => new TableController(url));

    const { tableData } = controller;

    const handlePaginationItemClick = (page: PaginationPage) => {
      controller.setCurrentPage(Number(page.caption)).fetch();
    };

    const theme = bunkerData?.data?.theme || 'clean';

    if (controller.errorMessage) {
      return <span>{controller.errorMessage}</span>;
    }

    if (!tableData || !bunkerData.isFetched) {
      if (controller.fetchingTableData && config.value.features.newDesignLift) {
        return (
          <Spin size="s" progress position="center" view="default" role="alert" aria-busy="true" />
        );
      }
      return null;
    }

    return (
      <Provider value={controller}>
        <div className={css.TableWithFilters}>
          {showExtendsFilters && <TemplateFilters sourceUrl={getFiltersUrl(controller.baseUrl)} />}
          <ThemeProvider value={theme}>
            <Table
              title={title}
              fetchingData={controller.fetchingTableData}
              withControls
              bunkerSettings={bunkerData?.data}
              updateTableAction={controller.fetch}
              tableData={tableData}
              pagination={tableData.gridPagination}
              paginationItemClick={handlePaginationItemClick}
            />
          </ThemeProvider>
        </div>
      </Provider>
    );
  },
);
