import { makeAutoObservable } from 'mobx';
import { jsonApiCall } from 'api/common';
import { FormAction } from 'types/api/form/Form';
import { Row } from 'components/Table/Table.types';
import { formatFormValuesToDataSet } from 'components/FormByScheme/utils/formatFormValuesToDataSet';
import { PartialBy } from 'utils/PartialBy';
import { concatUrlAndSearchParams } from 'utils/concatUrlAndSearchParams';
import { WithEditService } from '../../withEdit.service';
import { ActionResponse } from './ActionsManager.types';

class ActionsManager {
  private editServiceInstance: WithEditService;
  public actions: FormAction[] | undefined;
  public actionInProgress: FormAction | undefined;
  public updateTableAction: () => void | undefined;
  public currentAction: FormAction | undefined;
  public editingRowId: string | undefined;

  constructor(editServiceInstance: WithEditService) {
    this.editServiceInstance = editServiceInstance;
    makeAutoObservable(this, undefined, { autoBind: true });
  }

  public getCurrentAction() {
    return this.currentAction;
  }

  public setCurrentAction(action: FormAction | undefined) {
    this.currentAction = action;
  }

  public getEditingRowId() {
    return this.editingRowId;
  }

  public setEditingRowId(rowId: string | undefined) {
    this.editingRowId = rowId;
  }

  public prepareRowsToActionRequest(action: FormAction, values) {
    const editingRowId = this.editServiceInstance?.getEditingFormId();
    const row = formatFormValuesToDataSet(
      values,
      this.editServiceInstance.editingForm,
      editingRowId,
    );
    if (action.mode === 'Bulk') {
      const rows = [] as Row[];
      this.editServiceInstance.selectedRows?.forEach((id) => {
        rows.push({ ...row, id });
      });
      return rows;
    }
    return [row];
  }

  public cancelAction() {
    this.setCurrentAction(undefined);
    this.setEditingRowId(undefined);
    this.editServiceInstance.clearEditingForm();
  }

  public submitActionForm(values) {
    const currentAction = this.getCurrentAction();
    if (currentAction?.confirmCaption) {
      const isContinue = confirm(currentAction.confirmCaption);
      if (!isContinue) {
        return;
      }
    }
    if (currentAction) {
      const rows = this.prepareRowsToActionRequest(currentAction, values);
      if (currentAction.mode === 'Row') {
        const editingRowId = this.getEditingRowId();
        editingRowId &&
          this.doActionRequest(rows, { id: editingRowId })?.then(() => {
            this.editServiceInstance.clearEditingForm();
          });
        return;
      }
      this.doActionRequest(rows)?.then(() => {
        this.editServiceInstance.clearEditingForm();
      });
    }
  }

  public setUpdateTableAction(cb: () => void) {
    this.updateTableAction = cb;
  }

  public setActionCollection(actions?: FormAction[]) {
    if (actions) {
      this.actions = actions;
    }
  }

  public getActionById(id: string) {
    if (this.actions) {
      return this.actions.find((action) => action.id === id);
    }
  }

  public actionResponseHandler({ results }) {
    const action = this.getCurrentAction();
    if (!action) {
      return;
    }
    switch (action.refreshMode) {
      case 'ReloadGrid':
        this.updateTableAction?.();
        break;
      case 'ApplyResultData':
        const successRows = results.filter((row) => row.isSuccess);
        this.editServiceInstance.updateRowsByChangeType(successRows);
        break;
      default:
        break;
    }
    return results;
  }

  private afterActionRequest() {
    this.actionInProgress = undefined;
  }

  public doActionRequest(data?: PartialBy<Row, 'id'>[], queryParams?: Record<string, string>) {
    const action = this.getCurrentAction();
    if (!action) {
      return;
    }
    let dataBody = data && { args: data };
    this.actionInProgress = action;

    const url = action.customUrl || concatUrlAndSearchParams(action.actionUrl, queryParams);
    if (url) {
      return jsonApiCall<{ args: PartialBy<Row, 'id'>[] } | undefined, ActionResponse>({
        url,
        method: action.actionMethod || 'POST',
        data: dataBody,
      })
        .then(this.actionResponseHandler)
        .then((report) => {
          action.isReport &&
            this.editServiceInstance.setEditingReport({
              type: String(action.reportType),
              data: report,
            });
          this.setCurrentAction(undefined);
        })
        .finally(() => {
          this.afterActionRequest();
        });
    }
  }

  public checkConfirmation(action: FormAction): boolean {
    if (action.confirmCaption) {
      const isContinue = confirm(action.confirmCaption);
      if (!isContinue) {
        this.cancelAction();
        return false;
      }
    }
    return true;
  }

  public async bulkAction(action: FormAction) {
    if (action.isForm && action.actionUrl) {
      this.editServiceInstance.fetchAndSetEditingForm(action.actionUrl);
    } else {
      const tableData = this.editServiceInstance.stateManager.getTableData();
      const rows = tableData?.filter((row) => this.editServiceInstance.selectedRows?.has(row.id));
      if (this.checkConfirmation(action)) {
        this.doActionRequest(rows);
      }
    }
  }

  public async simpleAction(action: FormAction) {
    if (action.isForm && action.actionUrl) {
      this.editServiceInstance.fetchAndSetEditingForm(action.actionUrl);
    } else if (this.checkConfirmation(action)) {
      this.doActionRequest();
    }
  }

  public async rowAction(action: FormAction, rowId: string) {
    if (action.isForm) {
      this.setEditingRowId(rowId);
      const formSrcUrl =
        action.customUrl || concatUrlAndSearchParams(action.actionUrl, { id: rowId });
      formSrcUrl && this.editServiceInstance.fetchAndSetEditingForm(formSrcUrl);
    } else if (this.checkConfirmation(action)) {
      this.setEditingRowId(rowId);
      this.doActionRequest(undefined, { id: rowId });
    }
  }

  public doAction(actionId: string, rowId?: string, customUrl?: string) {
    let action = this.getActionById(actionId);
    if (customUrl && action) {
      action = { ...action, customUrl };
    }
    if (action) {
      this.setCurrentAction(action);
      switch (action.mode) {
        case 'Bulk':
          this.bulkAction(action);
          break;
        case 'Action':
          this.simpleAction(action);
          break;
        case 'Row':
          if (rowId) {
            this.rowAction(action, rowId);
          }
          break;
        default:
          break;
      }
    }
  }
}

export { ActionsManager };
