import { makeAutoObservable } from 'mobx';
import { Row, Table } from 'components/Table/Table.types';
import { WithEditService } from '../../withEdit.service';

class StateManager {
  private editServiceInstance: WithEditService;
  public table: Table;

  constructor(editServiceInstance: WithEditService) {
    this.editServiceInstance = editServiceInstance;
    makeAutoObservable(this, undefined, { autoBind: true });
  }

  public setTable = (table: Table) => {
    this.table = table;
    this.editServiceInstance.selectedRows =
      table.meta.actions?.find((action) => action.mode === 'Bulk') && table.data[0] && new Set();
    this.editServiceInstance.actionsManager.setActionCollection(table.meta.actions);
    if (table.meta.createAction) {
      const addAccess = table.meta.createAction.access;
      this.editServiceInstance.permissions = {
        ...this.editServiceInstance.permissions,
        add: { access: addAccess },
      };
    }
  };

  public addRowLocally = (newRow: Row): Row => {
    if (this.table) {
      const newTable = { ...this.table };
      if (newTable) {
        newTable.data.push(newRow);
        this.setTable(newTable);
      }
    }
    return newRow;
  };

  public updateRowLocally(newRow: Row): Row {
    const newTable = { ...this.table };
    if (newTable) {
      const rowIndex = newTable.data.findIndex((row) => row.id === newRow.id);
      newTable.data[rowIndex] = newRow;
      this.setTable(newTable);
    }
    return newRow;
  }

  public updateRowsLocally(rows: Row[]) {
    if (!Array.isArray(rows) || rows.length === 0) {
      return;
    }
    const newTable = { ...this.table };
    if (newTable) {
      rows.forEach((row) => {
        const index = newTable.data.findIndex((item) => item.id === row.id);
        if (index > -1 && newTable.data[index]) {
          newTable.data[index] = row;
        }
      });
      this.setTable(newTable);
    }
  }

  public removeRowLocally(rowId: string): void {
    if (this.table) {
      this.setTable({
        ...this.table,
        data: this.table.data.filter((row) => row.id !== rowId),
      });
    }
  }

  public getTable(): Table | undefined {
    return this.table;
  }

  public getTableData(): Table['data'] | undefined {
    return this.table?.data;
  }

  public getTableMeta(): Table['meta'] | undefined {
    return this.table?.meta;
  }
}

export { StateManager };
