import { makeAutoObservable } from 'mobx';
import { jsonApiCall } from 'api/common';
import { formatFormValuesToDataSet } from 'components/FormByScheme/utils/formatFormValuesToDataSet';
import { PartialBy } from 'utils/PartialBy';
import { AnyObject } from 'react-final-form';
import { ChangedField } from 'components/FinalForm/Form/Form.context';
import Bluebird from 'bluebird';
import { Form } from 'types/api/form/Form';
import isEqual from 'lodash/isEqual';
import { Row } from '../../Table.types';
import { EditingReport } from './withEdit.types';
import { prepareRowToFormData } from './utils';
import { ActionsManager } from './helpers/ActionsManager';
import { StateManager } from './helpers/StateManager';

interface Permission {
  access: number;
}

export class WithEditService {
  private baseUrl: string = '';
  private apiRequestPromise?: Bluebird<unknown>;
  public actionsManager: ActionsManager;
  public stateManager: StateManager;
  public editingReport: EditingReport | undefined;
  public selectedRows: Set<string> | undefined;
  public rowsInLoading: string[] = [];
  public editingForm: Form | undefined;
  public permissions: { [k: string]: Permission } = {};
  public formTitle: string | undefined = undefined;

  constructor(baseUrl?: string) {
    if (baseUrl) {
      this.baseUrl = baseUrl;
    }
    this.actionsManager = new ActionsManager(this);
    this.stateManager = new StateManager(this);
    makeAutoObservable(this, undefined, { autoBind: true });
  }

  public setBaseUrl(url: string): void {
    this.baseUrl = url;
  }

  public setFormTitle(title: string | undefined) {
    this.formTitle = title;
  }

  private oneRowEndpoint = (rowId: string): string => {
    const hasQueryParams = this.baseUrl.includes('?');
    if (hasQueryParams) {
      return this.baseUrl.replace('?', `/rows/${rowId}?`);
    }

    return `${this.baseUrl}/rows/${rowId}`;
  };

  private manyRowsEndpoint = (): string => {
    const hasQueryParams = this.baseUrl.includes('?');
    if (hasQueryParams) {
      return this.baseUrl.replace('?', '/rows?');
    }

    return `${this.baseUrl}/rows`;
  };

  private row = (rowId: string): Row | undefined => {
    const table = this.stateManager.getTable();
    if (!table) {
      return;
    }

    return table.data.find((row) => row.id === rowId);
  };

  public setEditingForm = (form: Form) => {
    this.editingForm = form;
  };

  public getEditingFormId(): string | undefined {
    return this.editingForm?.data[0]?.id;
  }

  public clearEditingForm() {
    this.editingForm = undefined;
  }

  public async fieldChangeHandler(changed: ChangedField, oldValues: AnyObject) {
    const fieldMeta = this.editingForm?.meta.fields.find(
      ({ type, id }) => type + id === changed.name,
    );
    if (fieldMeta?.isFieldsUpdateNeeded) {
      const values = { ...oldValues, [changed.name]: changed.value };
      await this.updateForm(values, oldValues);
    }
  }

  private async updateForm(values: AnyObject, oldValues: AnyObject) {
    const editingRowId = this.editingForm?.data[0]?.id;
    const fields = formatFormValuesToDataSet(values, this.editingForm, editingRowId);
    const oldFields = formatFormValuesToDataSet(oldValues, this.editingForm, editingRowId);

    let formUrl = this.baseUrl;
    if (this.actionsManager?.currentAction) {
      const { currentAction } = this.actionsManager;
      formUrl = currentAction?.customUrl || currentAction?.actionUrl || '';
    }
    if (!formUrl) return;
    const [path, query] = formUrl.split('?');
    const url = path + '/update-fields' + (query ? '?' + query : '');

    if (this.apiRequestPromise) {
      this.apiRequestPromise.cancel();
    }

    this.apiRequestPromise = jsonApiCall({
      url,
      data: {
        id: editingRowId,
        oldFields: oldFields.fields,
        fields: fields.fields,
      },
      global: false,
    }).then(this.setEditingForm);

    await this.apiRequestPromise;
  }

  public submitForm = (values) => {
    if (this.editingForm) {
      let row: PartialBy<Row, 'id'>;
      const rowId = this.editingForm && this.editingForm.data[0].id;
      if (rowId) {
        row = formatFormValuesToDataSet(values, this.editingForm, rowId);
        this.setRowLoading(rowId);
        return this.updateRow(row)
          .then(this.stateManager.updateRowLocally)
          .finally(() => {
            this.unsetRowLoading(rowId);
          });
      }
      row = formatFormValuesToDataSet(values, this.editingForm);
      return this.createRow(row).then(this.stateManager.addRowLocally);
    }
  };

  public async fetchAndSetEditingForm(url: string) {
    const form = await this.fetchEditingForm(url);
    if (form) {
      this.setFormTitle(form.meta.title);
      this.setEditingForm(form);
    }
  }

  public updateRowsByChangeType(successRows) {
    const rowsForAdd = successRows.filter((row) => row.changeType === 'Create');
    const rowsForDelete = successRows
      .filter((row) => row.changeType === 'Delete')
      .map((row) => row.id);
    const rowsForUpdate = successRows.filter((row) => row.changeType === 'Update');
    const table = this.stateManager.getTable();
    let newTableData = table?.data ? [...table.data] : [];

    if (rowsForAdd.length > 0) {
      newTableData.push(rowsForAdd[0].data);
    }
    if (rowsForUpdate.length > 0) {
      rowsForUpdate.forEach((row) => {
        const index = newTableData.findIndex((item) => item.id === row.id);
        if (index > -1 && newTableData[index]) {
          newTableData[index] = row.data;
        }
      });
    }
    if (rowsForDelete.length > 0) {
      newTableData = newTableData.filter((row) => !rowsForDelete.includes(row.id));
    }
    if (table) {
      this.stateManager.setTable({
        ...table,
        data: newTableData,
      });
    }
  }

  private setRowLoading = (rowId: string) => {
    if (!this.rowsInLoading.includes(rowId)) {
      this.rowsInLoading = [...this.rowsInLoading, rowId];
    }
  };

  private unsetRowLoading = (rowId: string) => {
    this.rowsInLoading = this.rowsInLoading.filter((id) => id !== rowId);
  };

  public addRow = async () => {
    const table = this.stateManager.getTable();
    const row = { fields: table?.meta.createAction!.defaultFields };
    const editingForm = prepareRowToFormData({ ...table, data: [row] });
    this.setFormTitle(editingForm.meta.title);
    this.setEditingForm(editingForm);
  };

  public createRow = async (row): Promise<Row> => {
    return jsonApiCall({
      url: this.manyRowsEndpoint(),
      method: 'POST',
      data: row,
    });
  };

  public async fetchEditingForm(url: string): Promise<Form | undefined> {
    return jsonApiCall({
      method: 'GET',
      url,
    });
  }

  public editRow = async (rowId: string): Promise<void> => {
    const editableRow = this.row(rowId);
    if (!editableRow) {
      return;
    }
    const table = this.stateManager.getTable();
    let editingForm = prepareRowToFormData({ ...table, data: [editableRow] });
    this.setEditingForm(editingForm);
    this.setFormTitle(editingForm.meta.title);
  };

  public saveRow = <Response = Row>(
    row: Row,
    url: string = this.oneRowEndpoint(row.id),
  ): Promise<Response> => {
    return jsonApiCall({
      url,
      method: 'POST',
      data: row,
    });
  };

  public updateRow = async (row): Promise<Row> => {
    return this.saveRow(row);
  };

  public removeRow = async (rowId: string): Promise<void> => {
    const table = this.stateManager.getTable();
    if (!table || !confirm('Удалить строку?')) {
      return;
    }

    this.setRowLoading(rowId);

    const row = this.row(rowId);
    return jsonApiCall({
      url: (row && row.url) || this.oneRowEndpoint(rowId),
      method: 'DELETE',
    })
      .then(() => {
        this.stateManager.removeRowLocally(rowId);
      })
      .finally(() => {
        this.unsetRowLoading(rowId);
      });
  };

  get tableRowIds() {
    const tableData = this.stateManager.getTableData();
    return tableData?.map((row) => row.id);
  }

  public selectRow(id: string) {
    if (this.selectedRows) {
      this.selectedRows.add(id);
    }
  }

  public unselectRow(id: string) {
    if (this.selectedRows && this.selectedRows.has(id)) {
      this.selectedRows.delete(id);
    }
  }

  public toggleSelectRow(id: string) {
    if (this.selectedRows) {
      this.selectedRows.has(id) ? this.unselectRow(id) : this.selectRow(id);
    }
  }

  public selectAllRows() {
    if (this.selectedRows) {
      this.selectedRows = new Set(this.tableRowIds);
    }
  }

  public unselectAllRows() {
    if (this.selectedRows) {
      this.selectedRows.clear();
    }
  }

  public setSelectedRows(rows: Set<string>) {
    this.selectedRows = rows;
  }

  public isAllRowsSelected() {
    if (this.selectedRows) {
      const rowIdsAsSet = new Set(this.tableRowIds);
      return isEqual(this.selectedRows, rowIdsAsSet);
    }
    return false;
  }

  public setEditingReport(report: EditingReport) {
    this.editingReport = report;
  }

  public clearEditingReport = () => {
    this.editingReport = undefined;
  };
}
