import React, { ComponentType, useEffect } from 'react';
import isEmpty from 'lodash/isEmpty';
import { observer } from 'mobx-react-lite';
import cx from 'classnames';
import { Text } from '@crm/components/dist/lego2/Text';
import { useEditingService } from './EditingContext';
import { EditPanel } from './components/EditPanel';
import { TableProps } from '../../Table.types';
import { WithEditProps } from './withEdit.types';
import { ModalForm } from './components/ModalForm';
import css from './withEdit.module.css';
import { withEditingProvider } from './withEditingProvider';
import { EditingReport } from './components/EditingReport';

export const withEdit = (WC: ComponentType<TableProps>) =>
  withEditingProvider(
    observer((props: WithEditProps & TableProps) => {
      const { showEditPanel = true, hasModal = true, baseUrl, updateTableAction, title } = props;

      const service = useEditingService();
      const { actionsManager } = service;

      if (updateTableAction) {
        actionsManager.setUpdateTableAction(updateTableAction);
      }

      useEffect(() => {
        if (props.tableData) {
          service.stateManager.setTable(props.tableData);
        }
      }, [props.tableData]);

      useEffect(() => {
        if (props.selectedRows) {
          service.setSelectedRows(props.selectedRows);
        }
      }, [props.selectedRows]);

      useEffect(() => {
        if (baseUrl) {
          service.setBaseUrl(baseUrl);
        }
      }, [baseUrl]);

      const table = service.stateManager.getTable();
      if (!table) {
        return null;
      }

      const isEditPanelShow =
        (!isEmpty(service.permissions) || actionsManager.actions) && showEditPanel;

      const className = cx(css.withEdit, {
        [css.withEdit__view_withEditPanel]: isEditPanelShow,
      });

      return (
        <div className={className}>
          <div
            className={cx(css.withEdit__topPanel_oneSection, {
              [css.withEdit__topPanel_twoSection]: isEditPanelShow && title,
            })}
          >
            {title && (
              <Text color="primary" weight="regular" typography="body-short-xl">
                {title}
              </Text>
            )}
            {isEditPanelShow && <EditPanel />}
          </div>
          <WC
            {...props}
            selectedRows={service.selectedRows}
            rowsInLoading={service.rowsInLoading}
            tableData={table}
          />
          {hasModal && <ModalForm />}
          <EditingReport />
        </div>
      );
    }),
  );
