import { makeAutoObservable } from 'mobx';
import { get } from 'api/common';
import { SortObject } from 'types/api/form/Form';
import { UserFilterAny } from 'types/api/form/UserFilter';
import { concatUrlAndSearchParams } from 'utils/concatUrlAndSearchParams';
import { Table } from '../../Table.types';
import { getUserFiltersWithData } from './utils/getUserFiltersWithData';

export class TableController {
  static defaultPageSize = 10;
  static defaultPage = 1;
  static defaultSort = [];
  static defaultTemplateFilters = [];
  static defaultUserFilters = {};

  private _baseUrl: string = '';
  private extendedQueryParams: Record<string, unknown> | undefined;
  private _tableData: Table | undefined;
  private _userFilters: Record<string, UserFilterAny> = TableController.defaultUserFilters;
  private currentPage: number = TableController.defaultPage;
  private pageSize: number = TableController.defaultPageSize;
  private _sort: SortObject[] = TableController.defaultSort;
  private _templateFilters: number[] = TableController.defaultTemplateFilters;
  private _fetchingTableData: boolean = false;
  private _errorMessage: string | undefined;

  constructor(baseUrl?: string) {
    makeAutoObservable(this, undefined, { autoBind: true });
    if (baseUrl) {
      this.setBaseUrl(baseUrl).fetch(baseUrl);
    }
  }

  get baseUrl() {
    return this._baseUrl;
  }
  set baseUrl(value) {
    this._baseUrl = value;
  }

  get tableData() {
    return this._tableData;
  }
  set tableData(value) {
    this._tableData = value;
  }

  get userFilters() {
    return this._userFilters;
  }
  set userFilters(value) {
    this._userFilters = value;
  }

  get sort() {
    return this._sort;
  }
  set sort(value) {
    this._sort = value;
  }

  get templateFilters() {
    return this._templateFilters;
  }
  set templateFilters(value) {
    this._templateFilters = value;
  }

  get fetchingTableData() {
    return this._fetchingTableData;
  }
  set fetchingTableData(value) {
    this._fetchingTableData = value;
  }

  get errorMessage() {
    return this._errorMessage;
  }
  set errorMessage(value) {
    this._errorMessage = value;
  }

  public findRowById(id: string) {
    return this.tableData?.data.find((el) => el.id === id);
  }

  public getSortById(columnId: string): SortObject | undefined {
    return this.sort.find((item) => item.id === columnId);
  }

  public getUserFilterById(columnId: string): UserFilterAny | undefined {
    return this.userFilters[columnId];
  }

  public setBaseUrl(url: string) {
    this.baseUrl = url;
    return this;
  }

  get baseUrlWithQuery() {
    let params = {
      query: JSON.stringify({
        templateFilters: this.templateFilters,
        userFilters: getUserFiltersWithData(this.userFilters),
        sort: this.sort,
        page: this.getCurrentPage(),
        pageSize: this.getPageSize(),
      }),
    };
    if (this.extendedQueryParams) {
      params = { ...this.extendedQueryParams, ...params };
    }
    return concatUrlAndSearchParams(this.baseUrl, params) || '';
  }

  public setExtendedQueryParams(queryParams: Record<string, unknown> | undefined) {
    this.extendedQueryParams = queryParams;
  }

  public fetch(url: string = this.baseUrlWithQuery) {
    this.fetchingTableData = true;
    this.errorMessage = undefined;
    return get({ url })
      .then(this.setTableData)
      .catch((error) => {
        this.errorMessage = error.message;
      })
      .finally(() => {
        this.fetchingTableData = false;
      });
  }

  public setTableData(table: Table) {
    this.tableData = table;
    this.userFilters = table?.filters?.userFilters || TableController.defaultUserFilters;
    this.templateFilters =
      table?.filters?.templateFilters || TableController.defaultTemplateFilters;
    this.sort = table?.sort || TableController.defaultSort;
    this.setCurrentPage(table?.gridPagination?.page || TableController.defaultPage);
    this.setPageSize(table?.gridPagination?.pageSize || TableController.defaultPageSize);
  }

  public addUserFilter(columnId: string, filter: UserFilterAny) {
    this.userFilters = { ...this.userFilters, [columnId]: filter };
    return this;
  }

  public removeUserFilter(columnId: string) {
    delete this.userFilters[columnId];
    return this;
  }

  // NOTICE: метод ориентирован на сортировку по 1 полю
  public addSort(sort: SortObject) {
    this.sort = [sort];
    return this;
  }

  public removeSort(columnId: string) {
    if (this.sort.find((item) => item.id === columnId)) {
      this.sort = [];
    }
    return this;
  }

  public addTemplateFilter(templateFilter: number) {
    this.templateFilters = [...this.templateFilters, Number(templateFilter)];
    return this;
  }

  public removeTemplateFilter(templateFilter: number) {
    this.templateFilters = this.templateFilters.filter((item) => item !== Number(templateFilter));
    return this;
  }

  public hasTemplateFilter(templateFilter: number) {
    return this.templateFilters.includes(Number(templateFilter));
  }

  public clearTemplateFilters() {
    this.templateFilters = TableController.defaultTemplateFilters;
    return this;
  }

  public toggleTemplateFilter(templateFilter: number) {
    const hasFilter = this.hasTemplateFilter(templateFilter);
    this.clearTemplateFilters();
    if (!hasFilter) {
      this.addTemplateFilter(templateFilter);
    }
    return this;
  }

  public getCurrentPage() {
    return this.currentPage;
  }

  public setCurrentPage(pageNum: number) {
    this.currentPage = pageNum;
    return this;
  }

  public getPageSize() {
    return this.pageSize;
  }

  public setPageSize(size: number) {
    this.pageSize = size;
    return this;
  }

  public resetCurrentPage() {
    this.currentPage = TableController.defaultPage;
    return this;
  }
}
