import React, { ReactElement, useCallback, useRef } from 'react';
import cx from 'classnames';
import Menu from '@crm/components/dist/lego2/Menu';
import { MixedItem } from '@yandex-lego/components/Menu';
import { Text } from '@crm/components/dist/lego2/Text';
import { useKeyboard } from 'utils/hooks/useKeyboard';
import { BubbleLike, TagCollectionProps } from './TagCollection.types';
import { bubbleToMenuItem } from './TagCollection.utils';
import css from './TagCollection.module.css';

export const TagCollection = <T extends BubbleLike>(
  props: TagCollectionProps<T>,
): ReactElement | null => {
  const { personal, common, creationText, onCreate, onChange, isFocused } = props;

  const menuRef = useRef<HTMLElement>(null);

  const emitCreate = useCallback(() => {
    onCreate?.(creationText!);
  }, [onCreate, creationText]);

  const handleChange = useCallback(
    (event: { target: { value: string | number } }) => {
      if (event.target.value === 'creation') {
        emitCreate();
        return;
      }

      if (onChange) {
        onChange(event.target.value);
      }
    },
    [onChange, emitCreate],
  );

  useKeyboard(
    {
      onSpaceCapture: (event) => event.stopImmediatePropagation(),
    },
    isFocused,
  );

  const handleActiveItemChange = (id: string) => {
    const menuNode = menuRef.current;
    if (!menuNode) {
      return;
    }

    if (id === 'creation') {
      setTimeout(() => {
        menuNode.scrollTop = menuNode.scrollHeight;
      }, 0);
      return;
    }

    if (id !== '0') {
      menuNode.scrollTop += 32;
    }
  };

  const items: MixedItem[] = [];
  if (common?.length) {
    items.push({
      items: common.map((item, index) => bubbleToMenuItem(item, index)),
    });
  }

  if (personal?.length) {
    items.push({
      title: 'Личные',
      items: personal.map((item, index) => bubbleToMenuItem(item, index + (common?.length || 0))),
    });
  }

  items.push({
    items: [
      {
        value: 'creation',
        id: 'creation',
        content: (
          <Text
            className={css.TagCollection__creationWrap}
            typography="body-long-m"
            weight="medium"
            align="start"
            overflow="ellipsis"
          >
            {creationText?.length ? `Создать «${creationText}»` : 'Создать метку'}
          </Text>
        ),
      },
    ],
  });

  return (
    <div className={cx(css.TagCollection, props.className)}>
      {Boolean(items.length) && (
        <Menu
          className={css.TagCollection__menu}
          items={items}
          onChange={handleChange}
          onActiveItemChange={handleActiveItemChange}
          innerRef={menuRef}
          focused={isFocused}
        />
      )}
    </div>
  );
};
