import React, { useState, FC, useEffect, RefObject, ChangeEvent } from 'react';
import Menu from '@crm/components/dist/lego2/Menu';
import Popup from '@crm/components/dist/lego2/Popup';
import { useIMask } from '@crm/react-imask';
import Textinput from '@crm/components/dist/lego2/Textinput';
import css from './TimeInput.module.css';
import { TimeInputProps } from './TimeInput.types';
import { getTimesList } from './TimeInput.utils';
import { maskConfig } from './TimeInput.config';

const timesList = getTimesList();

export const TimeInput: FC<TimeInputProps> = ({ onChange, value }) => {
  const [inputValue, setInputValue] = useState<string>(() => value || '');
  const { ref, maskRef } = useIMask(maskConfig);
  const mask = maskRef.current;

  const [popupVisibility, setPopupVisibility] = useState(false);

  const closeHandler = () => {
    setPopupVisibility(false);
  };

  const focusHandler = () => {
    mask?.updateValue();
    setPopupVisibility(true);
  };

  useEffect(() => {
    setInputValue(value || '');
    mask?.updateValue();
  }, [value, mask]);

  const changeHandler = (time: string) => {
    setInputValue(time);
    mask?.updateValue();
    if (mask?.masked?.isComplete) {
      onChange?.(time);
    }
    setPopupVisibility(false);
  };

  const setValueFromMenu = ({ target }: ChangeEvent<HTMLInputElement>) => {
    const { value } = target;
    changeHandler(value);
  };

  const blurHandler = () => {
    if (inputValue === '') {
      setInputValue(value || '');
      return;
    }

    if (mask && !mask.masked.isComplete) {
      setInputValue(value || '00:00');
      mask?.updateValue();
      return;
    }
  };

  return (
    <div data-testid="TimeInput">
      <Textinput
        placeholder="чч:мм"
        controlRef={ref as RefObject<HTMLInputElement>}
        onChange={changeHandler}
        onFocus={focusHandler}
        value={inputValue}
        onBlur={blurHandler}
      />
      <Popup
        anchor={ref}
        target="anchor"
        view="default"
        direction="bottom-start"
        visible={popupVisibility}
        onClose={closeHandler}
      >
        <Menu
          className={css.TimeInput__popup}
          width="max"
          items={timesList}
          onChange={setValueFromMenu}
        />
      </Popup>
    </div>
  );
};
