import React, { useState, useRef, useEffect } from 'react';
import Icon from '@crm/components/dist/lego2/Icon';
import Popup from '@crm/components/dist/lego2/Popup';
import { ToolTipWrapper } from '@crm/components/dist/lego2/Tooltip';
import cx from 'classnames';
import { TimelineItemStatus } from 'types/TimelineItemStatus';
import { StatusIconProps } from './StatusIcon.types';
import css from './StatusIcon.module.css';
import { statusIconProps, TOOLTIP_OPENDELAY } from './StatusIcon.config';
import { StatusIconAction, StatusIconActionProps } from './StatusIconAction';

export const StatusIcon: React.FC<StatusIconProps> = ({
  status,
  className,
  tooltip,
  actions = [],
}) => {
  const anchorRef = useRef<HTMLElement>(null);

  const [isPopupVisible, setPopupVisible] = useState(false);

  const supportedStatusIconProps = status && statusIconProps[status];
  const hasPopupActions = Boolean(actions.length);

  const showPopup = () => {
    if (hasPopupActions) {
      setPopupVisible(true);
    }
  };

  const hidePopup = () => {
    setPopupVisible(false);
  };

  const handleActionClick = (onClick?: StatusIconActionProps['onClick']) => (event) => {
    hidePopup();
    onClick?.(event);
  };

  useEffect(() => {
    hidePopup();
  }, [status]);

  if (!supportedStatusIconProps) {
    return null;
  }

  return (
    <>
      <ToolTipWrapper
        content={tooltip}
        direction="top-center"
        openDelay={TOOLTIP_OPENDELAY}
        view="inverse"
        className={css.StatusIcon__tooltip}
        disabled={isPopupVisible}
      >
        <span ref={anchorRef}>
          <Icon
            data-testid="status-icon"
            onClick={showPopup}
            className={cx(
              css.StatusIcon,
              {
                [css.StatusIcon_error]: status === TimelineItemStatus.Failed,
                [css.StatusIcon_clickable]: hasPopupActions,
              },
              className,
            )}
            {...supportedStatusIconProps}
          />
        </span>
      </ToolTipWrapper>
      <Popup
        data-testid="status-icon-popup"
        className={css.StatusIcon__popup}
        view="default"
        onClose={hidePopup}
        visible={isPopupVisible}
        anchor={anchorRef}
        target="anchor"
      >
        {actions.map(({ onClick, key, ...props }) => (
          <StatusIconAction key={key} {...props} onClick={handleActionClick(onClick)} />
        ))}
      </Popup>
    </>
  );
};
