import React, { FC, useState } from 'react';
import { FormattedTime } from 'react-intl';
import cx from 'classnames';
import Link from '@crm/components/dist/lego2/react-router/Link';
import Button, { ButtonProps } from '@crm/components/dist/lego2/Button';
import Icon from '@crm/components/dist/lego2/Icon';
import { RollerWithPopup } from 'components/Roller';
import { UpsaleFactors2 } from 'components/UpsaleFactors2';
import { PopupContext } from '@crm/components/dist/lego2/Popup/Popup';
import { UserLink } from 'components/UserLink';
import Toolbar from 'components/Toolbar';
import { TimelineItem2Props } from './TimelineItem2.types';
import css from './TimelineItem2.module.css';
import { StatusIcon, StatusIconProps } from './StatusIcon';

export const TimelineItem2: FC<TimelineItem2Props> = (props) => {
  const {
    className,
    children,
    upsaleFactors,
    date,
    dateUrl,
    author,
    mainActions,
    sideActions,
    statusActions,
    onClick,
    highlight,
    dateDisplayType = 'block',
    alignment = 'right',
    status,
    backgroundColor,
    addonBeforeDate,
  } = props;
  const [mainActionsVisible, setMainActionsVisible] = useState(false);

  const rootClassName = cx(
    css.TimelineItem2,
    css[`TimelineItem2_${String(alignment).toLowerCase()}`],
    className,
  );

  const containerClassName = cx(css.TimelineItem2__container, {
    [css.TimelineItem2__container_highlight]: highlight,
  });

  const dateClassName = cx(css.TimelineItem2__date, css[`TimelineItem2__date_${dateDisplayType}`]);

  const getStatusIcon = () => {
    if (!status) {
      return null;
    }

    const { tooltip, actions } = statusActions?.[status] || {};
    return (
      <StatusIcon
        status={status}
        tooltip={tooltip}
        actions={actions as StatusIconProps['actions']}
      />
    );
  };

  return (
    <div className={rootClassName}>
      {author && (
        <div className={css.TimelineItem2__authorAvatar}>
          <UserLink user={author} showName={false} avatarProps={{ size: 'customM' }} />
        </div>
      )}

      <div className={containerClassName} style={{ backgroundColor }} onClick={onClick}>
        {upsaleFactors && (
          <UpsaleFactors2
            className={css.TimelineItem2__upsale}
            upsaleFactors={upsaleFactors}
            alignment={alignment}
          />
        )}
        <div className={css.TimelineItem2__content}>
          <div className={css.TimelineItem2__preview}>{children}</div>
          <div className={dateClassName}>
            {addonBeforeDate}
            {/* eslint-disable-next-line @typescript-eslint/no-explicit-any */}
            <Link to={dateUrl as any} theme="secondary">
              <FormattedTime value={new Date(date)} hour12={false} />
            </Link>
            {getStatusIcon()}
          </div>
        </div>
        {Array.isArray(mainActions) && Boolean(mainActions.length) && (
          <Toolbar
            className={cx(css.TimelineItem2__mainActions, {
              [css.TimelineItem2__mainActions_visible]: mainActionsVisible,
            })}
          >
            <RollerWithPopup<ButtonProps>
              rollerClassNames={{
                containerClassName: css.TimelineItem2__mainActionsRoller,
              }}
              popupProps={{
                onVisibleChange: setMainActionsVisible,
                className: css.TimelineItem2__mainActionsPopup,
                containerClassName: css.TimelineItem2__mainActionsPopupContainer,
                element: ({ onClick, ...props }) => (
                  <PopupContext.Consumer>
                    {({ onClose }) => (
                      <Button
                        {...props}
                        onClick={(event) => {
                          onClose?.(event.nativeEvent, 'click');
                          onClick?.(event);
                        }}
                        view="clear"
                        className={css.TimelineItem2__mainActionsPopupItem}
                      />
                    )}
                  </PopupContext.Consumer>
                ),
                directions: ['bottom-center', 'top-center'],
                hasTail: true,
                mainOffset: 18,
              }}
              moreButtonProps={{
                element: (
                  <Button view="pseudo" className={css.TimelineItem2__mainActionsPopupButton}>
                    <Icon svg={'moreVert'} svgSize="l" />
                  </Button>
                ),
              }}
            >
              {mainActions.map((mainAction) => (
                <Button view="pseudo" {...mainAction} key={mainAction.key} />
              ))}
            </RollerWithPopup>
          </Toolbar>
        )}
      </div>

      {Array.isArray(sideActions) && Boolean(sideActions.length) && (
        <div
          className={cx(css.TimelineItem2__sideActions, {
            [css.TimelineItem2__sideActions_visible]: mainActionsVisible,
          })}
        >
          {sideActions.map(
            (sideAction) =>
              sideAction.render?.() ?? (
                <Button size="l" view="clear" {...sideAction} key={sideAction.key} />
              ),
          )}
        </div>
      )}
    </div>
  );
};
