import React, { FC, useState, useRef, useEffect, useMemo } from 'react';
import { TreeView } from 'components/TreeView';
import { Overlay } from 'components/Overlay';
import ComboInput from 'lego/components/ComboInput';
import Popup from '@crm/components/dist/lego2/Popup';
import { buildTreeDictionary } from 'components/TreeView/Tree/Tree.utils';
import { fetchTree } from './TreeInput.utils';
import css from './TreeInput.module.css';
import { TreeInputProps, TreeNodes } from './TreeInput.types';

export const TreeInput: FC<TreeInputProps> = (props) => {
  const { value, onChange, comboInputField = 'name', provider, endpoint, treeAdapter } = props;

  const [tree, setTree] = useState<TreeNodes>();

  const [isLoading, setIsLoading] = useState(false);
  const [visible, setVisible] = useState(false);
  const ref = useRef<HTMLElement>(null);

  useEffect(() => {
    setIsLoading(true);
    fetchTree((provider || endpoint)!)
      .then(setTree)
      .finally(() => setIsLoading(false));
  }, [provider, endpoint]);

  const items = useMemo(() => {
    if (tree) {
      return treeAdapter?.(tree) ?? tree;
    }
  }, [tree, treeAdapter]);

  const treeDictionary = useMemo(() => items && buildTreeDictionary(items), [items]);

  const togglePopup = () => {
    setVisible(!visible);
  };

  const handleClear = () => {
    setVisible(false);
    onChange(null);
  };

  const handleChange = (ids: string[]) => {
    const id = ids[0];
    onChange(treeDictionary[id]);
    setVisible(false);
  };

  const selected = useMemo(() => (value ? [value.id] : undefined), [value]);

  return (
    <>
      <ComboInput
        value={(value && value[comboInputField]) || undefined}
        size="m"
        view="default"
        onChange={togglePopup}
        onClear={handleClear}
        changeNode
        innerRef={ref}
      />
      <Popup
        className={css.TreeInput__Popup}
        visible={visible}
        target="anchor"
        anchor={ref}
        onClose={togglePopup}
        direction={['bottom-start', 'top-start']}
      >
        <Overlay hasSpinner display={isLoading} />
        {tree && (
          <TreeView
            items={items}
            showSearch
            singleSelect
            selected={selected}
            onChange={handleChange}
          />
        )}
      </Popup>
    </>
  );
};
