import React, { useCallback, useMemo } from 'react';
import cx from 'classnames';
import Icon from '@crm/components/dist/lego2/Icon/Icon';
import { TextSelection } from 'components/TextSelection';
import { ItemComponentProps, NodeData } from '../TreeView.types';
import { DEFAULT_LEFT_PADDING, NESTING_LEVEL_PADDING } from './Item.config';

import css from './Item.module.css';

export const Item = (props: ItemComponentProps<NodeData>) => {
  const {
    id,
    isLeaf,
    nestingLevel,
    data,
    getIsSelected,
    isExpanded,
    onExpand,
    onSelect,
    searchMatchRange,
    disableItemSelectionPredicate,
  } = props;

  const { label } = data;

  const style = useMemo(
    () => ({
      paddingLeft: `${DEFAULT_LEFT_PADDING + nestingLevel * NESTING_LEVEL_PADDING}px`,
    }),
    [nestingLevel],
  );

  const isSelected = useMemo(() => getIsSelected(id), [id, getIsSelected]);

  const isDisabled = useMemo(
    () => disableItemSelectionPredicate?.({ id, isLeaf, data, nestingLevel }),
    [data, disableItemSelectionPredicate, id, isLeaf, nestingLevel],
  );

  const handleSelect = useCallback(() => {
    if (!isDisabled) {
      onSelect(id, isLeaf);
    }
  }, [isDisabled, onSelect, id, isLeaf]);

  const handleExpand = useCallback(() => {
    onExpand(id);
  }, [id, onExpand]);

  return (
    <div data-testid="tree-view-item" className={css.Item} style={style}>
      {!isLeaf && (
        <span
          className={cx(css.Item__expandIcon, { [css.Item__expandIcon_rotated]: !isExpanded })}
          data-testid="tree-view-item-expand"
          onClick={handleExpand}
        >
          <Icon svg="arrowShort" svgSize="ms" />
        </span>
      )}
      <span onClick={handleSelect} className={css.Item__label} data-testid="tree-view-item-label">
        <TextSelection value={label} range={searchMatchRange} />
      </span>
      <Icon
        className={cx(css.Item__checkIcon, { [css.Item__checkIcon_hidden]: !isSelected })}
        svg="check"
        svgSize="ms"
      />
    </div>
  );
};
