import { Set } from 'immutable';

import { notNill } from 'utils/notNill/notNill';
import {
  FlatTreeNode,
  NodeData,
  Predicate,
  PredicateResult,
  TreeDictionary,
  TreeNode,
} from '../TreeView.types';

export function textPredicate<TData extends NodeData>(
  node: TreeNode<TData>,
  text: string,
): PredicateResult<TData> {
  const {
    data: { label },
  } = node;

  const index = label.toLowerCase().indexOf(text.toLowerCase());
  const isMatch = index !== -1;

  return {
    isMatch,
    newNode: {
      ...node,
      searchMatchRange: isMatch ? [index, index + text.length] : undefined,
    },
  };
}

export function filterTree<TData extends NodeData, TFilter>(
  rootNodes: TreeNode<TData>[],
  query: TFilter,
  predicate: Predicate<TData, TFilter>,
) {
  const filteredItemsIds: string[] = [];

  function filterBranch(node: TreeNode<TData>, isForceOpen: boolean = false) {
    const { items = [] } = node;

    filteredItemsIds.push(node.id);

    const { isMatch, newNode } = predicate(node, query);

    newNode.items = items
      ?.map((item) => filterBranch(item, isForceOpen || isMatch))
      .filter(notNill);

    if (isMatch || isForceOpen || newNode.items?.length) {
      return newNode;
    }

    return null;
  }

  const filteredItems = rootNodes.map((node) => filterBranch(node)).filter(notNill);

  return { filteredItems, filteredItemsIds };
}

export function traverseTree<TData extends NodeData>(
  rootNodes: TreeNode<TData>[],
  expandedNodes: Set<string>,
  isForceOpen?: boolean,
) {
  const stack = rootNodes.map((node) => ({ node, nestingLevel: 0 })).reverse();

  const result: FlatTreeNode<TData>[] = [];

  while (stack.length !== 0) {
    const node = stack.pop();

    if (!node) {
      continue;
    }

    const {
      node: { id, items = [] },
      nestingLevel,
    } = node;

    const isLeaf = items.length === 0;
    const isExpanded = Boolean(expandedNodes.has(id)) || isForceOpen;

    result.push({ isLeaf, nestingLevel, ...node.node, isExpanded });

    if (!isLeaf && isExpanded) {
      for (let i = items.length - 1; i >= 0; i--) {
        stack.push({
          nestingLevel: nestingLevel + 1,
          node: items[i],
        });
      }
    }
  }

  return result;
}

export function toggleNode(ids: Set<string>, id: string) {
  if (ids.has(id)) {
    return ids.delete(id);
  }
  return ids.add(id);
}

export function buildTreeDictionary<TData extends NodeData>(rootNodes: TreeNode<TData>[]) {
  let dictionary: TreeDictionary<TData> = {};

  const build = (node: TreeNode<TData>) => {
    const { id } = node;
    dictionary[id] = node;
    if (node?.items) {
      node.items.forEach((childNode) => build(childNode));
    }
  };

  rootNodes.forEach((node) => build(node));

  return dictionary;
}
