import { Set } from 'immutable';
import { useCallback, useEffect, useMemo, useState } from 'react';

import { useControlled } from 'utils/hooks/useControlled';
import { buildTreeDictionary, filterTree, toggleNode, traverseTree } from './Tree.utils';
import { NodeData, TreeViewComponentProps } from '../TreeView.types';

type useTreeProps<TData extends NodeData, TFilter> = Omit<
  TreeViewComponentProps<TData, TFilter>,
  | 'showSelectedBlock'
  | 'showSearch'
  | 'Buttons'
  | 'SearchComponent'
  | 'ItemComponent'
  | 'SelectedBlock'
  | 'disableItemSelectionPredicate'
> & { query?: TFilter };

function useTree<TData extends NodeData, TFilter>(props: useTreeProps<TData, TFilter>) {
  const {
    items,
    defaultSelected,
    defaultExpanded,
    singleSelect,
    onChange,
    onSave,
    onExpand,
    searchExpandBehavior = 'keepExpanded',
    query,
    search,
    shouldSelectBranch,
  } = props;

  const { predicate } = search;

  const [savedSelected, setSavedSelected] = useControlled<Set<string>>({
    value: props.selected ? Set(props.selected) : undefined,
    defaultValue: defaultSelected ? Set(defaultSelected) : Set([]),
  });
  const [currentSelected, setCurrentSelected] = useState<Set<string>>(
    props.selected ? Set(props.selected) : Set(defaultSelected ?? []),
  );

  useEffect(() => {
    setCurrentSelected(props.selected ? Set(props.selected) : Set(defaultSelected ?? []));
  }, [props.selected]);

  const [filteredItems, setFilteredItems] = useState(items);
  const [expanded, setExpanded] = useControlled<Set<string>>({
    value: props.expanded ? Set(props.expanded) : undefined,
    defaultValue: defaultExpanded ? Set(defaultExpanded) : Set([]),
  });

  const isForceOpen = useMemo(() => searchExpandBehavior === 'keepExpanded' && Boolean(query), [
    query,
    searchExpandBehavior,
  ]);

  const treeTraversal = useMemo(() => traverseTree(filteredItems, expanded, isForceOpen), [
    filteredItems,
    expanded,
    isForceOpen,
  ]);

  const treeDictionary = useMemo(() => buildTreeDictionary<TData>(items), [items]);

  const getNodeDataById = useCallback((id: string) => treeDictionary[id], [treeDictionary]);

  const handleChange = useCallback(
    (newSelected: Set<string>) => {
      if (newSelected !== undefined && newSelected !== currentSelected) {
        setCurrentSelected(newSelected);
        onChange?.(newSelected.toArray());
      }
    },
    [onChange, currentSelected],
  );

  const handleSave = useCallback(() => {
    if (savedSelected !== currentSelected) {
      setSavedSelected(currentSelected);
      onSave?.(currentSelected.toArray());
    }
  }, [onSave, savedSelected, currentSelected, setSavedSelected]);

  const handleSelectItem = useCallback(
    (id: string) => {
      if (singleSelect) {
        handleChange(currentSelected.has(id) ? Set() : Set([id]));
      }

      if (!singleSelect) {
        handleChange(toggleNode(currentSelected, id));
      }
    },
    [singleSelect, handleChange, currentSelected],
  );

  const handleSelectBranch = useCallback(
    (id) => {
      const branchRoot = getNodeDataById(id);
      const fullBranch = traverseTree([branchRoot], expanded, true).map(({ id }) => id);

      if (currentSelected.has(branchRoot.id)) {
        handleChange(currentSelected.subtract(fullBranch));
      } else {
        handleChange(currentSelected.union(fullBranch));
      }
    },
    [getNodeDataById, expanded, currentSelected, handleChange],
  );

  const handleSelectToggle = useMemo(
    () => (shouldSelectBranch ? handleSelectBranch : handleSelectItem),
    [handleSelectBranch, handleSelectItem, shouldSelectBranch],
  );

  const handleExpandToggle = useCallback(
    (id: string) => {
      if (!isForceOpen) {
        setExpanded(toggleNode(expanded, id));
        onExpand?.(toggleNode(expanded, id).toArray());
      }
    },
    [expanded, isForceOpen, onExpand, setExpanded],
  );

  const getIsSelected = useCallback((id: string) => currentSelected.has(id), [currentSelected]);

  useEffect(() => {
    if (query === undefined) {
      setFilteredItems(items);
      return;
    }

    const { filteredItems, filteredItemsIds } = filterTree(items, query, predicate);

    if (searchExpandBehavior === 'expandOnce' && Boolean(query)) {
      setExpanded(Set(filteredItemsIds));
    }

    setFilteredItems(filteredItems);
  }, [items, predicate, query, searchExpandBehavior, setExpanded]);

  return {
    currentSelected,
    savedSelected,
    treeTraversal,
    isForceOpen,
    handleSave,
    handleSelectToggle,
    handleExpandToggle,
    handleChange,
    getIsSelected,
    getNodeDataById,
    treeDictionary,
  };
}

export { useTree };
