import React, { useState } from 'react';
import Checkbox from '@crm/components/dist/lego2/Checkbox/Checkbox';
import Button from '@crm/components/dist/lego2/Button/Button';
import { Text } from '@crm/components/dist/lego2/Text';
import { TreeNode } from './TreeView.types';
import { buildTreeDictionary } from './Tree/Tree.utils';
import { TreeView } from './TreeView';
import { testData, TestNodeData } from './stubs/testData';
import { ButtonsClearAll } from './Buttons/ButtonsClear';
import Dropdown from '@crm/components/dist/lego2/Dropdown';
import BrowserScrollOverrider from '../BrowserScrollOverrider';

export default {
  title: 'TreeView',
  component: TreeView,
  argTypes: {
    onChange: { action: 'onChange' },
    onSave: { action: 'onSave' },
    searchExpandBehavior: {
      defaultValue: 'keepExpanded',
      control: {
        type: 'radio',
      },
      options: ['keepExpanded', 'expandOnce', 'none'],
    },
  },
  parameters: { actions: { argTypesRegex: '^on.*' } },
  decorators: [
    (Story) => (
      <div>
        <Story />
        <BrowserScrollOverrider />
      </div>
    ),
  ],
};

const ControlledWrapper = (props) => {
  const [selected, setSelected] = useState<string[]>(['4', '8', '9']);

  const handleSave = (ids) => {
    setSelected(ids);
  };

  return <TreeView {...props} selected={selected} onSave={handleSave} />;
};

const WithDropdown = (props) => {
  return (
    <Dropdown
      view="pseudo"
      hasArrow
      popupDirections={['bottom-right']}
      content={
        <div style={{ height: '532px', width: '280px' }}>
          <TreeView {...props} />
        </div>
      }
    >
      TreeView
    </Dropdown>
  );
};

const WithExpandWrapper = (props) => {
  const [expanded, setExpanded] = useState<string[]>([]);

  const treeDictionary = buildTreeDictionary(testData);
  const treeIds = Object.keys(treeDictionary);

  const handleExpand = (ids) => {
    setExpanded(ids);
  };

  const handleExpandAll = () => {
    setExpanded(treeIds);
  };

  const handleCollapseAll = () => {
    setExpanded([]);
  };

  return (
    <>
      <Button onClick={handleExpandAll}>Exapand all</Button>
      <Button onClick={handleCollapseAll}>Collapse all</Button>
      <TreeView {...props} expanded={expanded} onExpand={handleExpand} />
    </>
  );
};

const CustomSearchComponent = ({ query, setQuery }) => {
  return (
    <div style={{ padding: '19px 16px' }}>
      <Checkbox
        checked={query}
        onChange={() => {
          setQuery(!query);
        }}
      />
      <Text style={{ marginLeft: '16px' }}>filter by isArchived=true</Text>
    </div>
  );
};

const customPredicate = (node: TreeNode<TestNodeData>, query: boolean) => {
  const isMatch = query ? Boolean(node.data.isArchived) : true;
  const newNode = {
    ...node,
    data: {
      label: node.data.isArchived && query ? node.data.label + ' (is archived)' : node.data.label,
    },
  };

  return { isMatch, newNode };
};

const customDisableItemSelectionPredicate = (node: TreeNode<TestNodeData>) => {
  const {
    data: { isArchived },
  } = node;

  return !isArchived;
};

function generateBigTree(rootsCnt, childrenForRootCnt) {
  return new Array(rootsCnt).fill({}).map((item, ind) => ({
    id: ind,
    data: {
      label: `item${ind}`,
    },
    items: new Array(childrenForRootCnt).fill({}).map((item, innerInd) => ({
      id: ind * rootsCnt + innerInd + 1,
      data: { label: `item${ind * rootsCnt + innerInd + 1}` },
    })),
  }));
}

export const Simple = (args) => <TreeView {...args} />;
export const SimpleControlled = (args) => <ControlledWrapper {...args} />;
export const SearchReset = (args) => <TreeView {...args} />;
export const SearchResetControlled = (args) => <ControlledWrapper {...args} />;
export const SearchClear = (args) => <TreeView {...args} />;
export const SearchClearControlled = (args) => <ControlledWrapper {...args} />;
export const CustomSearch = (args) => <TreeView {...args} />;
export const WithExpandAll = (args) => <WithExpandWrapper {...args} />;
export const WithCustomDisableItemSelection = (args) => (
  <TreeView {...args} disableItemSelectionPredicate={customDisableItemSelectionPredicate} />
);
export const InDropdown = (args) => <WithDropdown {...args} />;

export const BigData = (args) => (
  <TreeView
    {...args}
    items={generateBigTree(args.bigDataProps.rootItems, args.bigDataProps.childrenForRoot)}
  />
);

const defaultArgs = {
  items: testData,
  showSearch: false,
  defaultSelected: undefined,
  singleSelect: false,
  shouldSelectBranch: false,
  showSelectedBlock: false,
  disableSelection: false,
  useCustomSearch: false,
};

Simple.args = {
  ...defaultArgs,
  defaultSelected: ['2', '3'],
};

SimpleControlled.args = {
  ...defaultArgs,
};

SearchReset.args = {
  ...defaultArgs,
  showButtons: true,
  showSearch: true,
  defaultSelected: ['2', '3'],
  showSelectedBlock: true,
};

SearchResetControlled.args = {
  ...defaultArgs,
  showButtons: true,
  showSearch: true,
  showSelectedBlock: true,
};

SearchClear.args = {
  ...defaultArgs,
  showSearch: true,
  showButtons: true,
  Buttons: ButtonsClearAll,
  defaultSelected: ['2', '3'],
  showSelectedBlock: true,
};

SearchClearControlled.args = {
  ...defaultArgs,
  showButtons: true,
  Buttons: ButtonsClearAll,
  showSearch: true,
  showSelectedBlock: true,
};

CustomSearch.args = {
  ...defaultArgs,
  showButtons: true,
  showSearch: true,
  useCustomSearch: true,
  SearchComponent: CustomSearchComponent,
  predicate: customPredicate,
};

WithExpandAll.args = {
  ...defaultArgs,
};

WithCustomDisableItemSelection.args = {
  ...defaultArgs,
};

InDropdown.args = {
  ...defaultArgs,
  showButtons: true,
  Buttons: ButtonsClearAll,
  showSearch: true,
  showSelectedBlock: true,
};

BigData.args = {
  ...defaultArgs,
  showButtons: true,
  Buttons: ButtonsClearAll,
  showSearch: true,
  showSelectedBlock: true,
  bigDataProps: {
    rootItems: 1000,
    childrenForRoot: 100,
  },
};

BigData.parameters = {
  bigDataProps: {
    rootItems: {
      type: 'number',
      childrenForRoot: 'number',
    },
  },
};
