import React, { useState } from 'react';
import { Divider } from '@yandex-lego/components/Divider';
import { useDebounce } from 'utils/hooks/useDebounce';
import { textPredicate } from './Tree/Tree.utils';
import { useTree } from './Tree/useTree';
import { Tree } from './Tree/Tree';
import { NodeData, TreeViewComponentProps, TreeViewProps } from './TreeView.types';
import css from './TreeView.module.css';
import { QUERY_DEBOUNCE_VALUE } from './TreeView.config';

const TreeViewComponent = <TData extends NodeData, TFilter>(
  props: TreeViewComponentProps<TData, TFilter>,
) => {
  const [queryState, setQueryState] = useState<TFilter>();

  const query = useDebounce(queryState, QUERY_DEBOUNCE_VALUE);

  const {
    showSelectedBlock,
    showSearch,
    showButtons,
    search,
    Buttons = Tree.Buttons,
    ItemComponent = Tree.Item,
    SelectedBlock = Tree.SelectedBlock,
    disableItemSelectionPredicate,
    ...restProps
  } = props;

  const { SearchComponent } = search;

  const {
    currentSelected,
    savedSelected,
    handleSave,
    handleSelectToggle,
    handleExpandToggle,
    handleChange,
    treeTraversal,
    getIsSelected,
    treeDictionary,
  } = useTree<TData, TFilter>({
    ...restProps,
    search,
    query,
  });

  return (
    <Tree>
      {showSearch && (
        <div className={css.TreeView__header}>
          <SearchComponent query={queryState} setQuery={setQueryState} />
        </div>
      )}
      <Tree.List
        ItemComponent={ItemComponent}
        onSelect={handleSelectToggle}
        onExpand={handleExpandToggle}
        treeTraversal={treeTraversal}
        getIsSelected={getIsSelected}
        disableItemSelectionPredicate={disableItemSelectionPredicate}
      />
      {((showSelectedBlock && Boolean(currentSelected.size)) || showButtons) && (
        <>
          <Divider className={css.TreeView__divider} />
          {showSelectedBlock && Boolean(currentSelected.size) && (
            <SelectedBlock
              selected={currentSelected}
              onSelectToggle={handleSelectToggle}
              treeDictionary={treeDictionary}
            />
          )}
          {showButtons && (
            <div className={css.TreeView__footer}>
              <Buttons
                onSave={handleSave}
                onChange={handleChange}
                savedSelected={savedSelected}
                currentSelected={currentSelected}
              />
            </div>
          )}
        </>
      )}
    </Tree>
  );
};

export const TreeView = <TData extends NodeData, TFilter>({
  search,
  ...restProps
}: TreeViewProps<TData, TFilter>) => {
  if (!search) {
    return (
      <TreeViewComponent
        search={{ SearchComponent: Tree.Search, predicate: textPredicate }}
        {...restProps}
      />
    );
  }

  return <TreeViewComponent search={search} {...restProps} />;
};
