import React, { useContext, useEffect, useState } from 'react';
import { mergeAllRefs } from '@yandex-lego/components/lib/mergeRefs';
import { EventAnchorContext } from './EventAnchorService';
import { observer } from 'mobx-react-lite';

export interface EventAnchorProps {
  innerRef?: React.Ref<HTMLElement>;
  visible?: boolean;
  scope?: React.RefObject<HTMLElement> | "inplace"
}

export const withEventAnchor = <TProps extends React.PropsWithChildren<EventAnchorProps>>(
  WrappedComponent: React.ComponentType<TProps>,
) => {
  return observer((props: TProps) => {
    const { innerRef, visible, scope } = props;

    const { setElement, removeElement } = useContext(EventAnchorContext) || {};

    const [instance, setInstance] = useState<HTMLElement | null>(null);

    useEffect(() => {
      if(instance && visible && scope !== 'inplace'){
        const requestAnimationFrameId = requestAnimationFrame(() => {
          setElement?.(instance)
        })

        return () => {
          cancelAnimationFrame(requestAnimationFrameId)
          requestAnimationFrame(() => {
            removeElement?.(instance)
          })
        }
      }
    }, [instance, visible, ])

    return (
      <WrappedComponent
        {...props}
        innerRef={mergeAllRefs(innerRef, setInstance)}
      />
    );
  });
};
