import React, { useEffect, useMemo } from 'react';
import throttle from 'lodash/throttle';
import { useInterval } from 'utils/hooks/useInterval';
import { config } from 'services/Config';
import { observer } from 'mobx-react-lite';
import { reportUxEvent, makeDataFromEvent, getEventAttributePath } from './UXLogger.utils';
import { LogEvent, CustomEventDetail, LogEventTarget } from './UXLogger.types';
import { useUXLogger } from './useUXLogger';
import {
  HOVER_THROTTLE,
  CUSTOM_EVENT_TYPE,
  ALIVE_INTERVAL,
  TESTID_NAME,
  UNSTTABLE_TESTID_NAME,
} from './UXLogger.config';
import {
  EventAnchorContext,
  EventAnchorService as EventAnchorServiceBase,
} from './EventAnchor/EventAnchorService';

export const UXLogger: React.FC = observer(({ children }) => {
  const EventAnchorService = useMemo(() => new EventAnchorServiceBase(), []);

  const getEventAdditionals = (event: LogEvent) => {
    const additionalPath = EventAnchorService.getAdditionalPath(
      event.composedPath() as LogEventTarget[],
    );

    const testIdPath = getEventAttributePath(additionalPath, TESTID_NAME);
    const unstableTestIdPath = getEventAttributePath(additionalPath, UNSTTABLE_TESTID_NAME);

    return { testIdPath, unstableTestIdPath };
  };

  const reportHoverEvent = throttle((event: LogEvent, path: LogEventTarget[]) => {
    const data = makeDataFromEvent(event);
    reportUxEvent({
      ...data,
      testIdPath: getEventAttributePath(path, TESTID_NAME),
      unstableTestIdPath: getEventAttributePath(path, UNSTTABLE_TESTID_NAME),
    });
  }, HOVER_THROTTLE);

  const handleHoverLog = (event: LogEvent) => {
    const additionalPath = EventAnchorService.getAdditionalPath(
      event.composedPath() as LogEventTarget[],
    );
    const path = event.composedPath().concat(additionalPath) as LogEventTarget[];
    reportHoverEvent(event, path);
  };

  const handleClickLog = (event: LogEvent) => {
    EventAnchorService.setAnchor(event);

    const additionals = getEventAdditionals(event);
    const data = makeDataFromEvent(event, additionals);

    reportUxEvent({ ...data, eventType: 'click' });
  };

  const handleCustomLog = (event: LogEvent) => {
    const { type, additionals } = event.detail as CustomEventDetail;

    if (type === 'click') {
      handleClickLog(event);
      return;
    }

    const data = makeDataFromEvent(event, additionals);
    reportUxEvent({ ...data, eventType: type });
  };

  const reportLoad = () => {
    const data = makeDataFromEvent(new Event('load') as LogEvent);
    reportUxEvent(data);
  };

  const reportAlive = () => {
    const data = makeDataFromEvent(new Event('alive') as LogEvent);
    reportUxEvent(data);
  };

  useUXLogger([
    { type: 'click', callback: handleClickLog },
    {
      type: 'visibilitychange',
      details: () => ({ visibilityState: document.visibilityState }),
      element: document,
    },
    { type: 'beforeunload' },
    { type: 'pointerover', callback: handleHoverLog },
    { type: CUSTOM_EVENT_TYPE, callback: handleCustomLog },
  ]);

  useInterval({
    callback: reportAlive,
    timeout: ALIVE_INTERVAL,
    isActive: config.value.features.newFrontendLogs,
  });

  useEffect(() => {
    if (config.value.features.newFrontendLogs) {
      reportLoad();
    }
    return () => {
      reportHoverEvent.cancel();
    };
  }, []);

  return (
    <EventAnchorContext.Provider value={EventAnchorService}>{children}</EventAnchorContext.Provider>
  );
});
