import React from 'react';
import { withRouter, RouteComponentProps } from 'react-router-dom';
import shallowEqual from 'utils/shallowEqual';
import { Filter, Parse, Format, FilterObj } from './Filter';

interface Props extends RouteComponentProps {
  filter: FilterObj;
  onFilterChange: (filter: FilterObj) => void;
  parse: Parse;
  format: Format;
  children?: React.ReactNode;
}

interface State {
  isInit: boolean;
}

class UrlFilterBinder extends React.Component<Props, State> {
  public static defaultProps = {
    parse: data => data,
    format: data => data,
    children: undefined,
  };

  private filter: Filter;

  public state = {
    isInit: false,
  };

  public constructor(props) {
    super(props);

    this.filter = new Filter({
      parse: this.props.parse,
      format: this.props.format,
      updateFilterObj: this.props.onFilterChange,
      updateUrl: this.updateUrl.bind(this),
    });
  }

  public componentDidMount() {
    this.update();
  }

  public componentDidUpdate(prevProps) {
    this.update(prevProps);
  }

  public setInit() {
    if (!this.state.isInit) {
      this.setState({ isInit: true });
    }
  }

  public update(prevProps?: Props) {
    if (
      (!prevProps && this.props.location.search) ||
      (prevProps && !this.filter.compareSearch(this.props.location.search))
    ) {
      this.filter.set(new URLSearchParams(this.props.location.search));
    } else if (
      (prevProps && !shallowEqual(this.props.filter, prevProps.filter)) ||
      (Object.keys(this.props.filter).length && !this.props.location.search)
    ) {
      this.filter.setFilter(this.props.filter);
    }

    this.setInit();
  }

  public updateUrl(search: string) {
    this.props.history.replace(`${this.props.location.pathname}${search}`);
  }

  public render() {
    const { children } = this.props;

    if (children && this.state.isInit) {
      return children;
    }

    return null;
  }
}

export default withRouter<Props, React.ComponentClass<Props, State>>(UrlFilterBinder);
