import React, { FC } from 'react';
import cx from 'classnames';
import Link from '@crm/components/dist/lego2/Link';
import Tooltip from 'lego/components/ToolTip';
import { Text } from '@crm/components/dist/lego2/Text';
import Avatar from 'components/Avatar';
import { createStaffUrl } from 'utils/createStaffUrl';
import { StaffCard } from 'components/StaffCard';
import { UserStatusIconByUser } from 'components/UserStatusIconByUser';
import { isExternalUser } from 'utils/isExternalUser';
import { UserLinkProps } from './UserLink.types';
import css from './UserLink.module.css';

export const UserLink: FC<UserLinkProps> = (props) => {
  const {
    user,
    onClick,
    isDisabled = false,
    allowBreakWord = false,
    isTooltipDisabled = false,
    showCrmPosition = false,
    showStatus = false,
    showName = true,
    avatarProps,
  } = props;

  const className = cx(css.UserLink, props.className, {
    [css.UserLink_hasStatus]: showStatus,
    [css.UserLink_withWrap]: allowBreakWord,
  });

  const renderedTooltipContent = (
    <>
      <Avatar
        theme="round"
        className={css.UserLink__avatar}
        size="s"
        {...avatarProps}
        user={user}
      />
      <Text className={css.UserLink__name} typography="control-m">
        {showName && user.name}
        {!isExternalUser(user) && showCrmPosition && ` (${user.crm_position})`}
        {!isExternalUser(user) && showStatus && (
          <UserStatusIconByUser userId={user.id} className={css.UserLink__status} view="mg" />
        )}
      </Text>
    </>
  );

  const renderedContent =
    user.login && !isTooltipDisabled ? (
      <Tooltip
        className={css.UserLink__tooltip}
        autoZIndex
        showDelay={300}
        hideDelay={300}
        hasTail={false}
        keepMounted
        tooltip={<StaffCard user={user} view="mg" />}
        stopPropagation
      >
        {renderedTooltipContent}
      </Tooltip>
    ) : (
      renderedTooltipContent
    );

  if (isDisabled) {
    return <div className={className}>{renderedContent}</div>;
  }

  return (
    <div className={className}>
      {user.login ? (
        <Link
          data-testid="user-link"
          className={css.UserLink__link}
          href={createStaffUrl(user.login)}
          tabIndex={-1}
          onClick={onClick}
          target="_blank"
        >
          {renderedContent}
        </Link>
      ) : (
        renderedContent
      )}
    </div>
  );
};
