import React, { useState, useEffect, useCallback, ReactNode } from 'react';
import { observer } from 'mobx-react-lite';
import Textinput from '@crm/components/dist/lego2/Textinput';
import Menu from '@crm/components/dist/lego2/Menu';
import Spin from '@crm/components/dist/lego2/Spin';
import { ErrorBlock } from 'components/ErrorBlock';
import { AsyncTaskStatus } from 'types/AsyncTaskStatus';
import Empty from 'components/Empty';
import { SkillAddMenuProps } from './SkillAddMenu.types';
import css from './SkillAddMenu.module.css';

export const SkillAddMenu = observer<SkillAddMenuProps>((props) => {
  const { size, onChange, userSkillsProviderFactory } = props;
  const [text, setText] = useState('');
  const [userSkillsProvider] = useState(() => userSkillsProviderFactory());

  useEffect(() => {
    userSkillsProvider.load();

    return () => {
      userSkillsProvider.destroy();
    };
  }, []);

  const handleChangeText = useCallback((text: string) => {
    setText(text);
    userSkillsProvider.load(text);
  }, []);

  const handleChange = useCallback(
    (event) => {
      if (!onChange) {
        return;
      }

      const id = Number(event.target.value);
      const skill = userSkillsProvider.items.find((item) => item.id === id);

      if (skill) {
        onChange(skill);
      }
    },
    [onChange],
  );

  let content: ReactNode = null;
  if (userSkillsProvider.status === AsyncTaskStatus.Pending) {
    content = <Spin progress view="default" size="m" />;
  } else if (userSkillsProvider.status === AsyncTaskStatus.Error) {
    content = <ErrorBlock error={userSkillsProvider.error} />;
  } else if (userSkillsProvider.status === AsyncTaskStatus.Complete) {
    if (!userSkillsProvider.items.length) {
      content = <Empty />;
    } else {
      content = (
        <Menu
          className={css.SkillAddMenu__menu}
          width="max"
          size={size as 's'}
          items={userSkillsProvider.items.map((item) => ({
            value: String(item.id),
            content: item.name,
          }))}
          onChange={handleChange}
        />
      );
    }
  }

  return (
    <div className={css.SkillAddMenu}>
      <div className={css.SkillAddMenu__searchWrap}>
        <Textinput
          placeholder="Поиск"
          autoFocus
          size={size}
          value={text}
          onChange={handleChangeText}
        />
      </div>
      <div className={css.SkillAddMenu__menuWrap}>{content}</div>
    </div>
  );
});
