import React, { FC, useEffect, useState, useRef } from 'react';
import { UserSkills } from 'components/UserSkills';
import { ArrayOperationBuilder } from 'utils/ArrayOperationBuilder';
import { UserSkillsInputProps } from './UserSkillsInput.types';
import { createSkillSetupFormModal } from './createSkillSetupFormModal';
import { SkillAdd } from './SkillAdd';
import { defaultUserSkillsProviderFactory } from './defaultUserSkillsProviderFactory';

export const UserSkillsInput: FC<UserSkillsInputProps> = ({
  value = [],
  onChange,
  size = 'xs',
  userSkillsProviderFactory = defaultUserSkillsProviderFactory,
  ...otherProps
}) => {
  const [modal] = useState(() => createSkillSetupFormModal());
  const [showAddPopup, setShowAddPopup] = useState(false);
  const buttonAddRef = useRef<HTMLElement | null>(null);

  useEffect(() => {
    modal.destroy();
  }, []);

  const handleDelete = (id) => {
    if (!onChange) {
      return;
    }

    const arrayOperationBuilder = new ArrayOperationBuilder(value);
    const removeOperation = arrayOperationBuilder.removeItemById(id);

    if (removeOperation) {
      onChange(removeOperation.array.next, removeOperation);
    }
  };

  const handleAddOrChangeSkill = (skill) => {
    let title = '';
    let initSkillValue = '1';
    if (skill.value) {
      title = `Редактирование навыка: ${skill.name}`;
      initSkillValue = String(skill.value);
    } else {
      title = `Добавить навык: ${skill.name}`;
    }

    modal.open({
      title,
      onSubmit: (values) => {
        if (!onChange) {
          return;
        }

        const newSkill = { ...skill, value: Number(values.skillValue) };

        const arrayOperationBuilder = new ArrayOperationBuilder(value);
        const changeOrAddOperation = arrayOperationBuilder.changeOrAddItem(newSkill);

        if (changeOrAddOperation) {
          onChange(changeOrAddOperation.array.next, changeOrAddOperation);
        }
      },
      initialValues: { SelectskillValue: initSkillValue },
    });
  };

  const handleChangeSkill = (id, item) => {
    handleAddOrChangeSkill(item);
  };

  return (
    <>
      <UserSkills
        value={value}
        size={size}
        {...otherProps}
        onDelete={handleDelete}
        onClick={handleChangeSkill}
        onAdd={() => setShowAddPopup(!showAddPopup)}
        buttonRef={buttonAddRef}
        placeholder="Добавить навык"
      />
      <SkillAdd
        visible={showAddPopup}
        setVisibility={setShowAddPopup}
        anchor={buttonAddRef}
        size={size}
        onAdd={handleAddOrChangeSkill}
        userSkillsProviderFactory={userSkillsProviderFactory}
      />
    </>
  );
};
