import { Subject, from, Subscription, EMPTY, ObservableInput } from 'rxjs';
import { debounceTime, distinctUntilChanged, switchMap, catchError, tap } from 'rxjs/operators';
import { makeAutoObservable, runInAction } from 'mobx';
import { UserSkill } from 'types/UserSkill';
import { AsyncTaskStatus } from 'types/AsyncTaskStatus';
import { logger } from 'services/Logger';
import { UserSkillsProvider } from './UserSkillsInput.types';

export type UserSkillsProviderCallback = (text: string) => ObservableInput<UserSkill[]>;

export class UserSkillsProviderByCallback implements UserSkillsProvider {
  status: AsyncTaskStatus = AsyncTaskStatus.Idle;
  items: UserSkill[] = [];
  error?: Error;

  private searchText$ = new Subject<string>();
  private subscription = new Subscription();

  constructor(private getUserSkills: UserSkillsProviderCallback) {
    makeAutoObservable(this);

    this.init();
  }

  destroy() {
    this.subscription.unsubscribe();
  }

  load(text: string = '') {
    this.searchText$.next(text);
  }

  private init() {
    this.subscription.add(
      this.searchText$
        .pipe(
          debounceTime(250),
          distinctUntilChanged(),
          tap(() => {
            runInAction(() => {
              this.status = AsyncTaskStatus.Pending;
            });
          }),
          switchMap((text) =>
            from(this.getUserSkills(text)).pipe(
              catchError((error) => {
                logger.reportAppErrorOnly(error);
                runInAction(() => {
                  this.error = error;
                  this.status = AsyncTaskStatus.Error;
                });

                return EMPTY;
              }),
            ),
          ),
        )
        .subscribe({
          next: (data) => {
            runInAction(() => {
              this.items = data;
              this.status = AsyncTaskStatus.Complete;
            });
          },
        }),
    );
  }
}
