import React, { useState } from 'react';
import { observer } from 'mobx-react-lite';
import { Virtuoso } from 'react-virtuoso';
import { VirtualListItem } from 'types/VirtualList';
import { AsyncTaskStatus } from 'types/AsyncTaskStatus';
import AutoSizer from 'react-virtualized-auto-sizer';
import Empty from 'components/Empty';
import { VirtualListServiceContext, VirtualListUIContext } from './context';
import { VirtualListProps } from './VirtualList.types';
import { Loader } from './Loader';
import { ErrorWithRetry } from './ErrorWithRetry';
import { NextLoader, PreviousLoader } from './EdgeLoader';
import { MAIN_SPINNER_TEST_ID } from './VirtualList.config';
import { ListWrapper } from './ListWrapper';

export const VirtualList = observer(
  <Item extends VirtualListItem>({
    listService,
    itemContent,
    className,
    style,
    ...other
  }: VirtualListProps<Item>) => {
    const [isScrolling, setIsScrolling] = useState(false);

    const handleScrolling = (value: boolean) => setIsScrolling(value);

    if (listService.loadInitTask.status === AsyncTaskStatus.Pending) {
      return <Loader data-testid={MAIN_SPINNER_TEST_ID} className={className} style={style} />;
    }

    if (listService.loadInitTask.status === AsyncTaskStatus.Error) {
      return (
        <ErrorWithRetry
          view="full"
          className={className}
          style={style}
          error={listService.loadInitTask.error || new Error('Неизвестная ошибка')}
          onRetry={listService.retryInit}
        />
      );
    }

    if (listService.loadInitTask.status === AsyncTaskStatus.Idle) {
      return null;
    }

    if (!listService.length) {
      return (
        <Empty
          className={className}
          style={style}
          text="Список пуст"
          icon="storage"
          mode="inline"
        />
      );
    }

    return (
      <div className={className} style={style}>
        <AutoSizer>
          {({ height, width }) => (
            <VirtualListServiceContext.Provider value={listService}>
              <VirtualListUIContext.Provider value={{ isScrolling }}>
                <Virtuoso
                  isScrolling={handleScrolling}
                  alignToBottom
                  style={{ height, width }}
                  firstItemIndex={listService.firstItemIndex}
                  totalCount={listService.length}
                  initialTopMostItemIndex={listService.initialTopMostItemIndex}
                  startReached={listService.startReached}
                  endReached={listService.endReached}
                  itemContent={(index) => itemContent(listService.getItemByAbsoluteIndex(index))}
                  computeItemKey={(index) => listService.getItemIdByAbsoluteIndex(index)}
                  components={{
                    Header: PreviousLoader,
                    Footer: NextLoader,
                    List: ListWrapper,
                  }}
                  followOutput="smooth"
                  increaseViewportBy={200}
                  {...other}
                />
              </VirtualListUIContext.Provider>
            </VirtualListServiceContext.Provider>
          )}
        </AutoSizer>
      </div>
    );
  },
);
