import React from 'react';
import cx from 'classnames';
import { logger } from 'services/Logger';
import Spinner from 'components/Spinner';
import { YandexFormLog } from './YandexFormLog';
import css from './YandexForm.module.css';
import { YandexFormProps } from './YandexForm.types';

/* cf docs https://wiki.yandex-team.ru/forms/instruction/iframe/ */

export default class YandexForm extends React.Component<YandexFormProps> {
  static defaultProps = {
    queryObject: {},
    minHeightTrigger: 10,
  };

  state = {
    height: undefined,
    isLoad: false,
  };

  frameRef = React.createRef<HTMLIFrameElement>();

  componentDidMount() {
    window.addEventListener('message', this.handleMessage);
    this.sendFormInfoReport('mount');
  }

  componentWillUnmount() {
    window.removeEventListener('message', this.handleMessage);
    this.sendFormInfoReport('unmount');
  }

  setInitialValuesToYandexForm = () => {
    const { queryObject } = this.props;

    if (this.frameRef.current && queryObject) {
      this.sendFormInfoReport('set initial values');
      Object.keys(queryObject).forEach((key) => {
        this.frameRef.current!.contentWindow!.postMessage(
          {
            message: 'set-question-value',
            slug: key,
            value: queryObject[key],
          },
          '*',
        );
      });
    }
  };

  handleMessage = (e) => {
    let { data } = e;
    const { source } = e;

    try {
      data = JSON.parse(data);
    } catch (error) {
      /* empty catch */
    }

    if (data.name === this.props.name) {
      if (data.message === 'ping') {
        source.postMessage('pong', '*');
      }

      if (data.type) {
        this.setInitialValuesToYandexForm();
        this.setState({ isLoad: true });
      }

      if (data.message === 'sent' && this.props.onSuccess) {
        this.sendFormInfoReport('success submitting');
        this.props.onSuccess();
      }

      const firstErrorPosition = data['iframe-offset'];
      if (firstErrorPosition && this.props.onFail) {
        this.props.onFail(firstErrorPosition);
      }

      const newHeight = data['iframe-height'];
      if (
        this.state.isLoad &&
        newHeight &&
        newHeight !== this.state.height &&
        newHeight > this.props.minHeightTrigger
      ) {
        this.setState({ height: newHeight });
      }
    }
  };

  private sendFormInfoReport(message: string) {
    logger.reportInfo(
      new YandexFormLog(message, { formId: this.props.formId, values: this.props.queryObject }),
    );
  }

  render() {
    const { formId, name, className, host, style } = this.props;
    const { height } = this.state;

    return (
      <Spinner overlay visible={!this.state.isLoad} className={cx(css.YandexForm, className)}>
        <iframe
          title="cf"
          ref={this.frameRef}
          className={cx(css.YandexForm__form)}
          style={{ height, visibility: height ? 'visible' : 'hidden', ...style }}
          src={`${host}surveys/${formId}/?iframe=1`}
          name={name}
        />
      </Spinner>
    );
  }
}
