import React from 'react';
import userEvent from '@testing-library/user-event';
import { render, screen } from '@testing-library/react';
import { Select } from './Select';

describe('Read/Select', () => {
  describe('props.label', () => {
    it('renders label', () => {
      render(<Select label="test label" />);
      const labelNode = screen.getByText('test label');

      expect(labelNode).toBeInTheDocument();
    });
  });

  describe('props.value', () => {
    describe('when defined', () => {
      it('renders value', () => {
        render(<Select label="test label" value="test value" />);
        const valueNode = screen.getByText('test value');

        expect(valueNode).toBeInTheDocument();
      });
    });

    describe('when undefined', () => {
      it('renders dash', () => {
        render(<Select label="test label" />);
        const dashNode = screen.getByText('—');

        expect(dashNode).toBeInTheDocument();
      });
    });
  });

  describe('props.isLoading', () => {
    describe('when is true', () => {
      it('renders loading overlay', () => {
        render(<Select label="test label" isLoading />);
        const overlayNode = screen.getByText('Сохранение изменений...');

        expect(overlayNode).toBeInTheDocument();
      });
    });

    describe('when is false', () => {
      it(`doesn't render loading overlay`, () => {
        render(<Select label="test label" isLoading={false} />);
        const overlayNode = screen.queryByText('Сохранение изменений...');

        expect(overlayNode).not.toBeInTheDocument();
      });
    });
  });

  describe('props.onClick', () => {
    it('calls on click', () => {
      const handleClick = jest.fn();
      render(<Select label="test label" access={3} onClick={handleClick} />);
      const labelNode = screen.getByText('test label');

      userEvent.click(labelNode);

      expect(handleClick).toBeCalled();
    });

    describe('when no access for editing', () => {
      it(`doesn't call on click`, () => {
        const handleClick = jest.fn();
        render(<Select label="test label" access={1} onClick={handleClick} />);
        const labelNode = screen.getByText('test label');

        userEvent.click(labelNode);

        expect(handleClick).not.toBeCalled();
      });
    });

    describe('when is loading', () => {
      it(`doesn't call on click`, () => {
        const handleClick = jest.fn();
        render(<Select label="test label" access={3} isLoading onClick={handleClick} />);
        const labelNode = screen.getByText('test label');

        userEvent.click(labelNode);

        expect(handleClick).not.toBeCalled();
      });
    });
  });

  describe('props.hasCarets', () => {
    describe('when true', () => {
      it('renders carets icon', () => {
        render(<Select label="test label" access={3} hasCarets />);

        expect(screen.getByLabelText('carets icon')).toBeInTheDocument();
      });
    });

    describe('when false', () => {
      it(`doesn't render carets icon`, () => {
        render(<Select label="test label" access={3} hasCarets={false} />);

        expect(screen.queryByLabelText('carets icon')).not.toBeInTheDocument();
      });
    });
  });
});
