import React, { FC, useState } from 'react';
import cx from 'classnames';
import Access from 'utils/Access';
import { Box } from 'components/design/ListItem/Box';
import { LeftRight } from 'components/design/ListItem/LeftRight';
import { Carets } from 'components/design/ListItem/Carets';
import { Text } from 'components/design/Text';
import { TextOverlay } from 'components/design/ListItem/TextOverlay';
import { groupClassName } from 'components/design/ListItem/groupClassName';
import { SelectProps } from './Select.types';
import css from './Select.module.css';

export const Select: FC<SelectProps> = (props) => {
  const {
    label,
    value,
    group,
    border = 'bottom-margin',
    access,
    isLoading,
    onClick,
    nodeRef,
    size,
    hasCarets = true,
  } = props;
  const [isHovering, setHovering] = useState(false);

  const isDisabled = !Access.isEdit(access);
  const isInteractive = !isDisabled && !isLoading;

  const cursor = isInteractive ? 'pointer' : 'default';
  const className = cx(css.Select, {
    [css.Select_hovered]: isHovering,
    [css.Select_disabled]: isDisabled,
  });

  const handleMouseEnter = () => {
    if (!isInteractive) {
      return;
    }

    setHovering(true);
  };

  const handleMouseLeave = () => {
    setHovering(false);
  };

  const handleClick = () => {
    if (isInteractive && onClick) {
      onClick();
    }
  };

  return (
    <Box
      className={className}
      border={border}
      group={group}
      onMouseEnter={handleMouseEnter}
      onMouseLeave={handleMouseLeave}
      onClick={handleClick}
      cursor={cursor}
      nodeRef={nodeRef}
    >
      <LeftRight
        size={size}
        left={<Text overflow="hidden">{label}</Text>}
        right={
          <Text
            className={css.Select__value}
            overflow="hidden"
            align="end"
            icon={(cls) => !isDisabled && hasCarets && <Carets className={cls} />}
          >
            {value ?? '—'}
          </Text>
        }
      />
      <TextOverlay display={isLoading} className={groupClassName(group)}>
        <Text>Сохранение изменений...</Text>
      </TextOverlay>
    </Box>
  );
};
