import React, { FC, ReactElement, RefObject, useRef } from 'react';
import { render, act } from '@testing-library/react';
import { Suggest as PureSuggest } from '../../Suggest';
import { withLoad } from './withLoad';

const RenderRef: FC<{ children: (ref: RefObject<HTMLDivElement>) => ReactElement }> = ({
  children,
}) => {
  const ref = useRef<HTMLDivElement>(null);

  return <div ref={ref}>{children(ref)}</div>;
};

describe('hoc/withLoad', () => {
  const requiredProps = {
    access: 3,
    label: 'label',
    onEditingStart: jest.fn(),
    onEditingStop: jest.fn(),
    onLoad: jest.fn(() => Promise.resolve([])),
  };

  const Suggest = withLoad(PureSuggest);

  describe('on mount', () => {
    it(`doesn't load items if isEditing = false`, () => {
      const handleLoad = jest.fn(() => Promise.resolve([]));

      render(
        <RenderRef>
          {(ref) => (
            <Suggest {...requiredProps} onLoad={handleLoad} isEditing={false} parentRef={ref} />
          )}
        </RenderRef>,
      );

      expect(handleLoad).not.toBeCalled();
    });

    it('loads items if isEditing = true', () => {
      const handleLoad = jest.fn(() => Promise.resolve([]));

      render(
        <RenderRef>
          {(ref) => <Suggest {...requiredProps} onLoad={handleLoad} isEditing parentRef={ref} />}
        </RenderRef>,
      );

      expect(handleLoad).toBeCalledTimes(1);
    });
  });

  describe('on text change', () => {
    it('loads items', () => {
      const handleLoad = jest.fn(() => Promise.resolve([]));

      const { rerender } = render(
        <RenderRef>
          {(ref) => (
            <Suggest {...requiredProps} text="" onLoad={handleLoad} isEditing parentRef={ref} />
          )}
        </RenderRef>,
      );

      rerender(
        <RenderRef>
          {(ref) => (
            <Suggest {...requiredProps} text="1" onLoad={handleLoad} isEditing parentRef={ref} />
          )}
        </RenderRef>,
      );

      expect(handleLoad).toBeCalledTimes(2);
    });

    it('uses debounce to load items', () => {
      const handleLoad = jest.fn(() => Promise.resolve([]));
      jest.useFakeTimers();

      const { rerender } = render(
        <RenderRef>
          {(ref) => (
            <Suggest
              {...requiredProps}
              debounce={500}
              text=""
              onLoad={handleLoad}
              isEditing
              parentRef={ref}
            />
          )}
        </RenderRef>,
      );

      rerender(
        <RenderRef>
          {(ref) => (
            <Suggest
              {...requiredProps}
              debounce={500}
              text="1"
              onLoad={handleLoad}
              isEditing
              parentRef={ref}
            />
          )}
        </RenderRef>,
      );

      expect(handleLoad).toBeCalledTimes(1);

      act(() => {
        jest.advanceTimersByTime(500);
      });

      expect(handleLoad).toBeCalledTimes(2);
    });
  });
});
