import React, { FC, useState, useEffect, useRef, MouseEvent } from 'react';
import { Step } from './Step';
import { TimeTooltip } from './TimeTooltip';
import { STEP_WIDTH, POPUP_OFFSET } from './Steps.constants';
import { StepsProps } from './Steps.types';
import css from './Steps.module.css';

export const Steps: FC<StepsProps> = (props) => {
  const { currentTime, duration, onCurrentTimeChange } = props;

  const nodeRef = useRef<HTMLDivElement>(null);
  const listRef = useRef<HTMLDivElement>(null);
  const [isTooltipVisible, setTooltipVisible] = useState(false);
  const [tooltipOffset, setTooltipOffset] = useState(0);
  const [tooltipTime, setTooltipTime] = useState(0);
  const [hoveredTo, setHoveredTo] = useState(-1);
  const [stepAmount, setStepAmount] = useState(10);

  useEffect(() => {
    const observer = new ResizeObserver((entries) => {
      const newStepAmount = Math.ceil(entries[0].target.clientWidth / (STEP_WIDTH * 2));
      setStepAmount(newStepAmount);
    });

    observer.observe(listRef.current!);
    return () => observer.disconnect();
  }, []);

  const handleMouseEnter = () => {
    setTooltipVisible(true);
  };

  const handleMouseMove = (event: MouseEvent<HTMLDivElement>) => {
    if (event.target === event.currentTarget) {
      return;
    }

    const target = event.target as HTMLElement;
    const parent = target.parentNode;
    const index = Array.prototype.indexOf.call(parent!.children, target);
    const fraction = (index + 1) / stepAmount;
    setTooltipTime(duration * fraction);
    setHoveredTo(index);
    setTooltipOffset(index * STEP_WIDTH * 2 - POPUP_OFFSET);
  };

  const handleMouseLeave = () => {
    setHoveredTo(-1);
    setTooltipVisible(false);
  };

  const handleClick = (event: MouseEvent<HTMLDivElement>) => {
    if (!onCurrentTimeChange || isNaN(duration)) {
      return;
    }

    let clickedStep = document.elementFromPoint(event.clientX, event.clientY) as HTMLDivElement;
    if (clickedStep === event.currentTarget) {
      clickedStep = document.elementFromPoint(
        event.clientX + STEP_WIDTH,
        event.clientY,
      ) as HTMLDivElement;
    }

    const parent = clickedStep.parentNode;
    const index = Array.prototype.indexOf.call(parent!.children, clickedStep);
    const fraction = (index + 1) / stepAmount;
    onCurrentTimeChange(duration * fraction);
  };

  const timePercent = Math.min(currentTime / Math.max(duration, 1), 1) * 100;
  let renderedSteps: React.ReactNode[] = [];
  for (let i = 1; i <= stepAmount; i++) {
    let state;
    if (i - 1 <= hoveredTo) {
      state = 'hovered';
    }

    if ((i / stepAmount) * 100 <= timePercent) {
      state = 'played';
    }

    const style = {
      width: STEP_WIDTH,
      marginRight: i === stepAmount ? 0 : STEP_WIDTH,
    };

    renderedSteps.push(<Step key={i} state={state} className={css.Steps__step} style={style} />);
  }

  return (
    <div className={css.Steps} ref={nodeRef}>
      <div
        className={css.Steps__list}
        ref={listRef}
        onClick={handleClick}
        onMouseEnter={handleMouseEnter}
        onMouseMove={handleMouseMove}
        onMouseLeave={handleMouseLeave}
      >
        {renderedSteps}
      </div>
      <TimeTooltip
        isVisible={isTooltipVisible}
        time={tooltipTime}
        scope={nodeRef}
        anchor={listRef}
        horizontalOffset={tooltipOffset}
      />
    </div>
  );
};
