import { useEffect, useState } from 'react';

export interface AudioState {
  isLoading: boolean;

  isPlaying: boolean;
  onPlayChange: (isPlaying: boolean) => void;

  rate: number;
  onRateChange: (rate: number) => void;

  currentTime: number;
  onCurrentTimeChange: (currentTime: number) => void;

  duration: number;
}

export const useAudioState = (audio: HTMLAudioElement): AudioState => {
  const [isLoading, setLoading] = useState(audio.readyState !== 4);
  const [isPlaying, setPlaying] = useState(!audio.paused);
  const [currentTime, setCurrentTime] = useState(audio.currentTime);
  const [rate, setRate] = useState(audio.playbackRate);
  const [duration, setDuration] = useState(audio.duration);

  useEffect(() => {
    const handleLoadedData = () => {
      setLoading(false);
    };

    const handlePlay = () => {
      setPlaying(true);
    };

    const handlePause = () => {
      setPlaying(false);
    };

    const handleEnded = () => {
      setPlaying(false);
    };

    const handleTimeUpdate = () => {
      setCurrentTime(audio.currentTime);
    };

    const handleRateChange = () => {
      setRate(audio.playbackRate);
    };

    const handleDurationChange = () => {
      setDuration(audio.duration);
    };

    audio.addEventListener('loadeddata', handleLoadedData);
    audio.addEventListener('play', handlePlay);
    audio.addEventListener('pause', handlePause);
    audio.addEventListener('timeupdate', handleTimeUpdate);
    audio.addEventListener('ratechange', handleRateChange);
    audio.addEventListener('durationchange', handleDurationChange);
    audio.addEventListener('ended', handleEnded);

    return () => {
      audio.removeEventListener('loadeddata', handleLoadedData);
      audio.removeEventListener('play', handlePlay);
      audio.removeEventListener('pause', handlePause);
      audio.removeEventListener('timeupdate', handleTimeUpdate);
      audio.removeEventListener('ratechange', handleRateChange);
      audio.removeEventListener('durationchange', handleDurationChange);
      audio.removeEventListener('ended', handleEnded);
    };
  }, [audio]);

  const handlePlayChange = (value: boolean) => {
    if (value) {
      audio.play();
    } else {
      audio.pause();
    }
    setPlaying(value);
  };

  const handleCurrentTimeChange = (value: number) => {
    audio.currentTime = value;
    setCurrentTime(value);
  };

  const handleRateChange = (value: number) => {
    audio.playbackRate = value;
    setRate(value);
  };

  return {
    isLoading,

    isPlaying,
    onPlayChange: handlePlayChange,

    currentTime,
    onCurrentTimeChange: handleCurrentTimeChange,

    rate,
    onRateChange: handleRateChange,

    duration,
  };
};
