import React, { FC, MouseEvent } from 'react';
import cx from 'classnames';
import Button from '@crm/components/dist/lego2/Button';
import Icon from '@crm/components/dist/lego2/Icon';
import { Box } from '../ListItem/Box';
import { Checkbox } from '../Checkbox';
import { Text } from '../Text';
import { CategoryProps } from './Category.types';
import css from './Category.module.css';
import {
  categoryTestId,
  checkboxTestId,
  expandingArrowTestId,
  transitionButtonTestId,
} from './Category.constants';

export const Category: FC<CategoryProps> = (props) => {
  const {
    className: outerClassName,
    group,
    border: outerBorder,
    canClick = true,
    onClick,
    isHighlighted,
    canCheck = true,
    onCheck,
    isChecked,
    canDoubleClickChecked = true,
    onDoubleClick,
    content: outerContent,
    hasExpandingArrow = false,
    hasTransitionButton = false,
    onTransitionClick,
    ...restProps
  } = props;

  const className = cx(css.Category, outerClassName, {
    [css.Category__highlighted]: isHighlighted,
  });

  const border = outerBorder ?? 'bottom';
  const cursor = canClick ? 'pointer' : 'default';
  const checkboxColor = isHighlighted ? '#fff' : undefined;
  let content = outerContent;
  if (typeof content === 'string') {
    content = <Text>{content}</Text>;
  }

  const handleClick = () => {
    if (canClick && onClick) {
      onClick();
    }
  };

  const handleDoubleClick = (event: MouseEvent<HTMLDivElement>) => {
    if (canDoubleClickChecked && canCheck && onCheck) {
      onCheck(!isChecked);
    }

    if (onDoubleClick) {
      onDoubleClick(event);
    }
  };

  return (
    <Box
      className={className}
      group={group}
      border={border}
      cursor={cursor}
      onClick={handleClick}
      onDoubleClick={handleDoubleClick}
      data-testid={categoryTestId}
      aria-selected={isHighlighted}
      {...restProps}
    >
      {canCheck && (
        <Checkbox
          className={css.Category__checkbox}
          checked={isChecked}
          onChange={onCheck}
          color={checkboxColor}
          data-testid={checkboxTestId}
        />
      )}
      <div className={css.Category__content}>{content}</div>
      {hasExpandingArrow && (
        <Icon
          className={css.Category__expandingArrow}
          glyph="type-arrow"
          type="arrow"
          direction="right"
          data-testid={expandingArrowTestId}
        />
      )}
      {hasTransitionButton && (
        <Button
          onClick={onTransitionClick}
          className={css.Category__transitionButton}
          view="clear"
          icon={(cls) => (
            <Icon svgSize="s" className={cx(cls, css.Category__transitionButtonIcon)} svg="arrow" />
          )}
          data-testid={transitionButtonTestId}
        />
      )}
    </Box>
  );
};
