import React, { FC, useState, useEffect } from 'react';
import cx from 'classnames';
import { FilterProps } from './Filter.types';
import { Box } from '../ListItem/Box';
import { Chevron } from '../ListItem/Chevron';
import { Text } from '../Text';
import css from './Filter.module.css';
import { getLabelStyle } from './Filter.utils';

export const Filter: FC<FilterProps> = (props) => {
  const {
    label,
    showValue = true,
    value,
    depth = 1,
    isDisabled = false,
    isPressed = false,
    isExpanded = false,
    canExpand = false,
    onClick,
    onExpand,
    border: outerBorder,
    group,
    ...restProps
  } = props;

  const [isHovering, setHovering] = useState(false);
  const cursor = isDisabled ? 'default' : 'pointer';
  const border = outerBorder ?? 'none';

  const className = cx(css.Filter, {
    [css.Filter_disabled]: isDisabled,
    [css.Filter_hovered]: isHovering,
    [css.Filter_pressed]: isPressed,
  });

  useEffect(() => {
    if (isDisabled) {
      setHovering(false);
    }
  }, [isDisabled]);

  const handleMouseEnter = () => {
    if (isDisabled) {
      return;
    }

    setHovering(true);
  };

  const handleMouseLeave = () => {
    setHovering(false);
  };

  const handleBoxClick = () => {
    if (isDisabled || !onClick) {
      return;
    }

    onClick();
  };

  const handleValueClick = () => {
    if (isDisabled || !onExpand || !canExpand) {
      return;
    }

    onExpand();
  };

  return (
    <Box
      title={label}
      className={className}
      border={border}
      group={group}
      onMouseEnter={handleMouseEnter}
      onMouseLeave={handleMouseLeave}
      cursor={cursor}
      onClick={handleBoxClick}
      {...restProps}
    >
      <Text
        aria-label="left"
        style={getLabelStyle(depth)}
        overflow="hidden"
        className={cx(css.Filter__label)}
      >
        {label}
      </Text>

      <Text
        aria-label="right"
        align="end"
        className={cx(css.Filter__value)}
        icon={(cls) =>
          canExpand && <Chevron className={cls} direction={isExpanded ? 'bottom' : 'right'} />
        }
        onClick={handleValueClick}
      >
        {showValue ? value : ''}
      </Text>
    </Box>
  );
};
