import React, { PureComponent } from 'react';
import throttle from 'lodash/throttle';
import { ItemsListProps, ItemsListState } from './ItemsList.types';
import { computeGroup, computeBorder } from '../Group.utils';
import { scrollToChildIndex } from './ItemsList.utils';
import { KEYBOARD_NAVIGATION_THROTTLE_MS } from './ItemsList.constants';
import { defaultGetValue } from '../Group.utils';

export class ItemsList extends PureComponent<ItemsListProps, ItemsListState> {
  public static defaultProps = {
    isFocused: false,
    scrollNode: null,
    isScrollable: false,
    items: [],
    getValue: defaultGetValue,
  };

  public constructor(props: ItemsListProps) {
    super(props);
    this.state = {
      keyboardHoverIndex: -1,
      mouseHoverIndex: -1,
    };
  }

  private handleMouseEnter = (index: number) => {
    this.setState({
      mouseHoverIndex: index,
    });
  };

  private handleMouseLeave = () => {
    this.setState({
      mouseHoverIndex: -1,
    });
  };

  private handleClick = (item: unknown) => {
    const { onChange } = this.props;

    if (onChange) {
      onChange(item);
    }
  };

  private handleKeyDown = throttle((event: KeyboardEvent) => {
    const { items, onChange } = this.props;

    if (!['ArrowUp', 'ArrowDown', 'Enter'].includes(event.key)) {
      return;
    }

    event.preventDefault();
    if (event.key === 'ArrowDown') {
      this.moveKeyboardHover('down', () => this.scrollToIndex(this.state.keyboardHoverIndex));
    }

    if (event.key === 'ArrowUp') {
      this.moveKeyboardHover('up', () => this.scrollToIndex(this.state.keyboardHoverIndex));
    }

    if (event.key === 'Enter' && onChange && this.state.keyboardHoverIndex !== -1) {
      onChange(items[this.state.keyboardHoverIndex]);
    }
  }, KEYBOARD_NAVIGATION_THROTTLE_MS);

  private moveKeyboardHover(direction: 'down' | 'up', callback?: () => void) {
    const { items } = this.props;

    if (direction === 'up') {
      this.setState((prev) => {
        if (prev.keyboardHoverIndex <= 0) {
          return {
            keyboardHoverIndex: items.length - 1,
          };
        }

        return {
          keyboardHoverIndex: prev.keyboardHoverIndex - 1,
        };
      }, callback);
    }

    if (direction === 'down') {
      this.setState((prev) => {
        if (prev.keyboardHoverIndex >= items.length - 1) {
          return {
            keyboardHoverIndex: 0,
          };
        }

        return {
          keyboardHoverIndex: prev.keyboardHoverIndex + 1,
        };
      }, callback);
    }
  }

  private subscribeOnKeyboardEvents() {
    document.addEventListener('keydown', this.handleKeyDown);
  }

  private unsubscribeFromKeyboardEvents() {
    document.removeEventListener('keydown', this.handleKeyDown);
  }

  private resetHoveredIndexes() {
    this.setState({
      keyboardHoverIndex: -1,
      mouseHoverIndex: -1,
    });
  }

  private hoverKeyboardFirstItem(callback?: () => void) {
    this.setState(
      {
        keyboardHoverIndex: 0,
      },
      callback,
    );
  }

  private hoverKeyboardSelectedItem(callback?: () => void) {
    const newIndex = Math.max(
      this.props.items.findIndex((item) => this.props.value === this.props.getValue(item)),
      0,
    );
    this.setState(
      {
        keyboardHoverIndex: newIndex,
      },
      callback,
    );
  }

  private hoverFirstOrSelectedItem(callback?: () => void) {
    if (this.props.value == null) {
      this.hoverKeyboardFirstItem(callback);
    } else {
      this.hoverKeyboardSelectedItem(callback);
    }
  }

  private scrollToIndex(index: number) {
    scrollToChildIndex(this.props.scrollNode, index);
  }

  public componentDidMount() {
    this.hoverFirstOrSelectedItem(() => this.scrollToIndex(this.state.keyboardHoverIndex));
    if (this.props.isFocused) {
      this.subscribeOnKeyboardEvents();
    }
  }

  public componentDidUpdate(prevProps: ItemsListProps) {
    if (!prevProps.isFocused && this.props.isFocused) {
      this.subscribeOnKeyboardEvents();
      this.hoverFirstOrSelectedItem(() => this.scrollToIndex(this.state.keyboardHoverIndex));
    }

    if (prevProps.isFocused && !this.props.isFocused) {
      this.unsubscribeFromKeyboardEvents();
      this.resetHoveredIndexes();
    }
  }

  public componentWillUnmount() {
    this.resetHoveredIndexes();
    if (this.props.isFocused) {
      this.unsubscribeFromKeyboardEvents();
    }
  }

  public render() {
    const { items, getValue, isScrollable, renderItem: ItemRender, value } = this.props;
    const { keyboardHoverIndex, mouseHoverIndex } = this.state;

    return (
      <>
        {items.map((item, index) => (
          <ItemRender
            key={getValue(item)}
            item={item}
            isHovered={[keyboardHoverIndex, mouseHoverIndex].includes(index)}
            isSelected={getValue(item) === value && value !== undefined}
            onClick={() => this.handleClick(item)}
            onMouseEnter={() => this.handleMouseEnter(index)}
            onMouseLeave={this.handleMouseLeave}
            group={computeGroup({
              length: items.length,
              isScrollable,
              index,
            })}
            border={computeBorder({
              length: items.length,
              index,
            })}
          />
        ))}
      </>
    );
  }
}
