import { Observable } from 'rxjs';
import { useState, useMemo, useEffect } from 'react';

export interface UseObservableOptions<T> {
  defaultNext?: T;
  defaultError?: Error;
  defaultComplete?: boolean;
}

export const useObservable = <T>(
  observable: Observable<T>,
  options: UseObservableOptions<T> = {},
) => {
  const [next, setNext] = useState<T | undefined>(options.defaultNext);
  const [error, setError] = useState<Error | undefined>(options.defaultError);
  const [complete, setComplete] = useState(options.defaultComplete ?? false);

  useEffect(() => {
    setNext(options.defaultNext);
    setError(options.defaultError);
    setComplete(options.defaultComplete ?? false);

    const subscription = observable.subscribe({
      next: setNext,
      error: setError,
      complete: () => setComplete(true),
    });

    return () => subscription.unsubscribe();
  }, [observable]);

  const result: [T | undefined, Error | undefined, boolean] = [next, error, complete];
  return useMemo(
    () => ({
      next,
      error,
      complete,
    }),
    result,
  );
};
