import React from 'react';
import PropTypes from 'prop-types';
import InfiniteList, { WithNavigationInfiniteList } from 'components/InfiniteList';
import get from 'lodash/get';
import Checkbox from '@crm/components/dist/lego2/Checkbox';
import theme from './style.modules.scss';
import HeaderCell from './HeaderCell';
import DefaultRow from './Row';

const indexCellStyle = {
  maxWidth: '50px',
  textAlign: 'right',
  color: '#999999',
};

const checkedCellStyle = {
  maxWidth: '50px',
  color: '#999999',
  alignItems: 'center',
  justifyContent: 'center',
};

export default class TableView extends React.Component {
  calculateColumnsStyles = () => {
    let columnsStyle;
    if (this.props.grid && this.props.grid.columns) {
      columnsStyle = {};
      columnsStyle.index = indexCellStyle;
      columnsStyle.checked = checkedCellStyle;
      this.props.grid.columns.forEach((column) => {
        const { name } = column;
        columnsStyle[name] = {
          flexGrow:
            this.props.flexRatio && (this.props.flexRatio[name] || this.props.flexRatio.default),
        };
      });
    }

    this.columnsStyle = columnsStyle;
  };

  handleItemClick(id, item, e) {
    if (this.props.onItemClick) {
      this.props.onItemClick(id, item, e);
    }
  }

  handleItemDoubleClick(id, item, e) {
    if (this.props.onItemDoubleClick) {
      this.props.onItemDoubleClick(id, item, e);
    }
  }

  renderHeader = () => {
    let renderColumns = [];
    if (this.props.grid) {
      renderColumns = this.props.grid.columns
        .map(
          (column) =>
            get(this.props, `columnVisibility.${column.name}`) && (
              <HeaderCell
                key={column.name}
                {...column}
                onClick={this.props.onSortChange}
                sortType={
                  (this.props.sort &&
                    this.props.sort.fieldName === column.name &&
                    this.props.sort.type) ||
                  undefined
                }
                theme={theme}
                style={this.columnsStyle && this.columnsStyle[column.name]}
                sortable={this.props.canSortColumn && column.sortable}
              />
            ),
        )
        .filter((column) => column);
    }

    const { onHeaderCheck, headerChecked } = this.props;

    return (
      Boolean(renderColumns.length) && (
        <div className={theme.headerWrap}>
          <div className={theme.header}>
            {onHeaderCheck && (
              <HeaderCell
                title={
                  <Checkbox
                    checked={headerChecked}
                    onChange={() => onHeaderCheck(!headerChecked)}
                  />
                }
                style={this.columnsStyle.checked}
              />
            )}
            <HeaderCell title="#" style={this.columnsStyle.index} />
            {renderColumns}
          </div>
        </div>
      )
    );
  };

  render() {
    if (!this.props.showIfEmpty && !this.props.isLoad) {
      return null;
    }

    const {
      className,
      rowComponent: Row,
      withNavigation,
      itemKey,
      headerChecked,
      itemsChecked,
      onItemCheck,
      ...rest
    } = this.props;

    this.calculateColumnsStyles();

    const listItems = this.props.items || [];

    const InfiniteComponent = withNavigation ? WithNavigationInfiniteList : InfiniteList;

    return (
      <InfiniteComponent
        selected={this.props.selectedId}
        className={className}
        header={this.renderHeader()}
        {...rest}
      >
        {listItems.map((item, index) => {
          let id;
          if (typeof itemKey === 'function') {
            id = itemKey(item);
          } else if (typeof itemKey === 'string') {
            id = item[itemKey];
          }

          id = id || index;

          const rowClassName =
            (id === this.props.selectedId && theme.row_selected) ||
            ((this.props.onItemClick || this.props.canSelect) && theme.row_canSelect) ||
            theme.row_default;

          return (
            <Row
              key={id}
              index={index + 1}
              id={id}
              item={item}
              onClick={(e) => this.handleItemClick(id, item, e)}
              onDoubleClick={(e) => this.handleItemDoubleClick(id, item, e)}
              className={rowClassName}
              cellClassName={theme.cell_body}
              columns={this.props.grid.columns.filter((column) =>
                get(this.props, `columnVisibility.${column.name}`),
              )}
              columnsStyle={this.columnsStyle}
              canSelect={Boolean(this.props.onItemClick) || this.props.canSelect}
              isChecked={headerChecked ? !itemsChecked?.has(item.id) : itemsChecked?.has(item.id)}
              onCheck={onItemCheck && (() => onItemCheck(item.id))}
            />
          );
        })}
      </InfiniteComponent>
    );
  }
}

TableView.propTypes = {
  name: PropTypes.string.isRequired,
  className: PropTypes.string,
  isLoad: PropTypes.bool,
  isLoading: PropTypes.bool,
  items: PropTypes.arrayOf(PropTypes.object),
  itemKey: PropTypes.oneOfType([PropTypes.string, PropTypes.func]),
  sort: PropTypes.shape({
    fieldName: PropTypes.string,
    type: PropTypes.string,
  }),
  grid: PropTypes.shape({
    columns: PropTypes.arrayOf(
      PropTypes.shape({
        name: PropTypes.string,
        title: PropTypes.string,
        sortable: PropTypes.bool,
      }),
    ),
    gridVersion: PropTypes.string,
  }),
  columnVisibility: PropTypes.objectOf(PropTypes.bool),
  flexRatio: PropTypes.objectOf(PropTypes.number),
  selectedId: PropTypes.oneOfType([PropTypes.number, PropTypes.string]),
  onItemClick: PropTypes.func,
  onItemDoubleClick: PropTypes.func,
  canSelect: PropTypes.bool,
  canSortColumn: PropTypes.bool,
  showIfEmpty: PropTypes.bool,
  withNavigation: PropTypes.bool,
  rowComponent: PropTypes.func,
  onSortChange: PropTypes.func,
  changeColumns: PropTypes.func,
};

TableView.defaultProps = {
  className: undefined,
  isLoad: false,
  isLoading: false,
  items: undefined,
  itemKey: 'id',
  sort: undefined,
  grid: undefined,
  onItemClick: undefined,
  onItemDoubleClick: undefined,
  columnVisibility: {},
  selectedId: undefined,
  flexRatio: {
    id: 1,
    default: 2,
  },
  canSelect: false,
  canSortColumn: true,
  showIfEmpty: false,
  rowComponent: DefaultRow,
  onSortChange: undefined,
  changeColumns: undefined,
  withNavigation: true,
};
