import { Component, createElement } from 'react';
import PropTypes from 'prop-types';
import { connect } from 'react-redux';
import { stateSaver } from 'modules/localStorage';
import { getFormValues } from 'redux-form';
import { bindActionCreators } from 'redux';
import get from 'lodash/get';
import forOwn from 'lodash/forOwn';
import mapValues from 'lodash/mapValues';
import hoistStatics from 'hoist-non-react-statics';
import getDisplayName from 'utils/getDisplayName';
import * as actions from './actions';
import interval from './components/interval';
import saveUI from './saveUI';

const getDisplayColumnString = obj => {
  if (!obj) {
    return undefined;
  }

  const resultArr = [];
  forOwn(obj, (value, key) => {
    if (value) {
      resultArr.push(key);
    }
  });

  return resultArr.join(',');
};

const createInfiniteList = initialConfig => {
  const config = {
    ...initialConfig,
  };
  return _WrappedComponent => {
    const WrappedComponent = interval(_WrappedComponent);

    class InfiniteList extends Component {
      static propTypes = {
        name: PropTypes.string.isRequired,
        forceDisplayAllColumns: PropTypes.bool,
        isFetch: PropTypes.bool,
        eof: PropTypes.bool,
        initialUpdateInterval: PropTypes.number,
        loadOnMount: PropTypes.bool,
        triggerLoad: PropTypes.bool,
        triggerReset: PropTypes.bool,
        reloadOnLoadUpdate: PropTypes.bool,
        saveUI: PropTypes.oneOfType([PropTypes.bool, PropTypes.object]),
        setUpdateInterval: PropTypes.func.isRequired,
        setResetData: PropTypes.func.isRequired,
        resetState: PropTypes.func.isRequired,
        onLoad: PropTypes.func.isRequired,
        clearLoad: PropTypes.func.isRequired,
        initialize: PropTypes.func.isRequired,
        request: PropTypes.func.isRequired,
        receive: PropTypes.func.isRequired,
        failed: PropTypes.func.isRequired,
        reload: PropTypes.func.isRequired,
        resetData: PropTypes.func.isRequired,
        refreshData: PropTypes.func.isRequired,
        sortChange: PropTypes.func.isRequired,
        load: PropTypes.func.isRequired,
        destroy: PropTypes.func.isRequired,
        refresh: PropTypes.func.isRequired,
        settings: PropTypes.instanceOf(Object),
        autoUpdate: PropTypes.bool,
        updateInterval: PropTypes.number,
        children: PropTypes.oneOfType([PropTypes.node, PropTypes.func]),
      };

      static defaultProps = {
        saveUI: undefined,
        forceDisplayAllColumns: false,
        initialUpdateInterval: undefined,
        loadOnMount: undefined,
        triggerLoad: undefined,
        triggerReset: undefined,
        reloadOnLoadUpdate: undefined,
        settings: undefined,
        isFetch: undefined,
        eof: undefined,
        children: undefined,
        autoUpdate: undefined,
        updateInterval: undefined,
      };

      constructor(props) {
        super(props);

        if (props.saveUI) {
          stateSaver.addHandler(saveUI(props.name));
        }
      }

      componentWillMount() {
        this.props.initialize({
          saveUI: this.props.saveUI,
          forceDisplayAllColumns: this.props.forceDisplayAllColumns,
        });
        this.setUpdateIntevalIfNeeded();
      }

      componentDidMount() {
        if (this.props.loadOnMount) {
          this.props.load();
        }
      }

      componentWillReceiveProps(nextProps) {
        this.setUpdateIntevalIfNeeded(nextProps);
        this.reloadIfNeeded(nextProps);
        this.isReset = this.resetIfNeeded(nextProps);
        this.loadIfNeeded(nextProps);
        // this.isSettingChange(nextProps);
      }

      componentWillUnmount() {
        this.props.destroy();
        this.cancelPromise();
      }

      setUpdateIntevalIfNeeded(nextProps) {
        if (nextProps) {
          if (nextProps.initialUpdateInterval !== this.props.initialUpdateInterval) {
            this.props.setUpdateInterval(nextProps.initialUpdateInterval);
          }
        } else {
          this.props.setUpdateInterval(this.props.initialUpdateInterval);
        }
      }

      cancelPromise = () => {
        if (this.loadResult) {
          this.loadResult.cancel();
        }
      };

      loadIfNeeded(nextProps) {
        const { clearLoad, triggerLoad } = this.props;
        if (!triggerLoad && nextProps.triggerLoad) {
          clearLoad();
          this.load(nextProps);
        }
      }

      reloadIfNeeded(nextProps) {
        if (this.props.reloadOnLoadUpdate && nextProps.onLoad !== this.props.onLoad) {
          this.props.reload();
        }
      }

      isSettingChange(nextProps) {
        if (this.props.settings !== undefined && this.props.settings !== nextProps.settings) {
          this.props.load();
          this.props.setResetData();
        }
      }

      resetIfNeeded(nextProps) {
        const { triggerReset, resetState } = this.props;
        if (!triggerReset && nextProps.triggerReset) {
          resetState();
          return true;
        }

        return false;
      }

      load(nextProps) {
        const {
          onLoad,
          request,
          receive,
          failed,
          resetState, // eslint-disable-line no-unused-vars
          resetData,
          refreshData,
        } = this.props;

        this.cancelPromise();

        const length = get(nextProps, 'items.length');
        const offset =
          nextProps.triggerRefresh || this.isReset || nextProps.triggerResetData ? 0 : length;

        request({ isLoadingFirstPage: !offset });
        this.loadResult = onLoad({
          offset,
          length: nextProps.triggerRefresh && length,
          sortField: nextProps.sort.fieldName,
          sortType: nextProps.sort.type,
          gridVersion: get(nextProps, 'grid.gridVersion'),
          columns: getDisplayColumnString(get(nextProps, 'columnVisibility')),
          ...nextProps.extraFetchParam,
        })
          .then(data => {
            if (nextProps.triggerRefresh) {
              refreshData(data);
            } else {
              if (nextProps.triggerResetData) {
                if (this.scrollNode) {
                  this.scrollNode.scrollTop = 0;
                }
                resetData();
              }

              receive(data);
            }

            this.isReset = false;
            this.loadResult = null;

            return data;
          })
          .catch(error => {
            if (nextProps.triggerRefresh) {
              failed(error, false);
            } else {
              if (nextProps.triggerResetData) {
                if (this.scrollNode) {
                  this.scrollNode.scrollTop = 0;
                }
                resetData();
              }

              failed(error, true);
            }

            this.loadResult = null;
            this.isReset = false;

            throw error;
          });

        return this.loadResult;
      }

      render() {
        const propsToPass = {
          ...this.props,
          onLoad: this.props.load,
          isLoading: this.props.isFetch,
          isEof: this.props.eof,
          onSortChange: this.props.sortChange,
          children: this.props.children,
          getScrollNode: node => {
            this.scrollNode = node;
          },
          callback: this.props.refresh,
          delay: this.props.autoUpdate && this.props.updateInterval,
        };

        delete propsToPass.sortChange;
        delete propsToPass.isFetch;
        delete propsToPass.triggerResetData;
        delete propsToPass.triggerReset;
        delete propsToPass.triggerLoad;
        delete propsToPass.extraFetchParam;
        delete propsToPass.eof;
        delete propsToPass.updateInterval;
        delete propsToPass.theme;
        delete propsToPass.triggerRefresh;
        delete propsToPass.load;
        delete propsToPass.update;
        delete propsToPass.reset;
        delete propsToPass.request;
        delete propsToPass.receive;
        delete propsToPass.failed;
        delete propsToPass.destroy;
        delete propsToPass.initialize;
        delete propsToPass.refresh;
        delete propsToPass.reload;
        delete propsToPass.autoUpdate;
        delete propsToPass.forceDisplayAllColumns;
        delete propsToPass.saveUI;
        delete propsToPass.itemUpdate;
        delete propsToPass.stopUpdate;
        delete propsToPass.startUpdate;
        delete propsToPass.setUpdateInterval;
        delete propsToPass.clearResetData;
        delete propsToPass.triggerSortChange;
        delete propsToPass.refreshData;
        delete propsToPass.resetData;
        delete propsToPass.resetState;
        delete propsToPass.setResetData;
        delete propsToPass.clearLoad;
        delete propsToPass.setExtraFetchParam;
        delete propsToPass.loadOnMount;
        delete propsToPass.reloadOnLoadUpdate;
        delete propsToPass.initialUpdateInterval;

        // хак только для списка в тикетах
        delete propsToPass.takeNextAvailableCount;

        return createElement(WrappedComponent, propsToPass);
      }
    }

    InfiniteList.displayName = `InfiniteList(${getDisplayName(WrappedComponent)})`;
    InfiniteList.WrappedComponent = WrappedComponent;

    const connector = connect(
      (state, props) => {
        const propsSettings = {};
        if (props.autoUpdate) {
          propsSettings.autoUpdate = props.autoUpdate;
        }

        const stateProps = {
          ...state.infiniteList[props.name],
          settings: props.settings ? props.settings : getFormValues(props.name)(state),
          ...propsSettings,
        };

        if (props.columnVisibility) {
          stateProps.columnVisibility = props.columnVisibility;
        }

        return stateProps;
      },
      (dispatch, initialProps) => {
        const bindList = actionCreator => actionCreator.bind(null, initialProps.name);
        const boundListACs = mapValues(actions, bindList);
        const connectedBoundListACs = bindActionCreators(boundListACs, dispatch);
        return () => ({
          ...connectedBoundListACs,
        });
      },
      undefined,
      { forwardRef: true },
    );

    const ConnectedInfiniteList = hoistStatics(connector(InfiniteList), WrappedComponent);
    ConnectedInfiniteList.defaultProps = config;

    return ConnectedInfiniteList;
  };
};

export default createInfiniteList;
