import Immutable from 'immutable';
import get from 'lodash/get';
import * as actionTypes from './actionTypes';
import ItemReducer from './ItemReducer';

const INIT_STATE = {
  isLoad: false,
  sort: {},
  settings: {},
  extraFetchParam: {},
  updateInterval: 0,
  autoUpdate: true,
};

const SORT_CHANGE_STATE = {
  asc: 'desc',
  desc: undefined,
  undefined: 'asc',
};

const getNewSortType = (state, fieldName) => {
  if (get(state, 'sort.fieldName') === fieldName) {
    return SORT_CHANGE_STATE[get(state, 'sort.type')];
  }

  return 'asc';
};

const behaviors = {
  [actionTypes.INITIALIZE](state, { payload }) {
    const newState = {
      ...INIT_STATE,
      ...state,
    };

    return {
      ...newState,
      settings: {
        ...newState.settings,
        ...payload,
      },
    };
  },
  [actionTypes.LOAD](state, { reset }) {
    const update = { triggerLoad: true };
    if (reset) {
      update.triggerResetData = true;
    }

    return {
      ...state,
      ...update,
    };
  },
  [actionTypes.UPDATE](state, { updater, isRaw }) {
    if (isRaw) {
      return updater(state);
    }

    const result = updater(Immutable.fromJS(state));
    if (result && typeof result.toJS === 'function') {
      return result.toJS();
    }
    return result;
  },
  [actionTypes.CLEAR_LOAD](state) {
    const newState = { ...state };
    delete newState.triggerLoad;

    return newState;
  },
  [actionTypes.REQUEST](state, { data = {} }) {
    return { ...state, isFetch: true, isLoadingFirstPage: data.isLoadingFirstPage };
  },
  [actionTypes.RECEIVE](state, { data }) {
    const currentItems = state.items || [];
    const newItems = data.items || [];

    let newState = {
      ...state,
      ...data,
      items: [...currentItems, ...newItems],
      isLoad: true,
    };
    delete newState.isFetch;
    delete newState.isError;

    const columns = get(data, 'grid.columns');
    const forceDisplayAllColumns = get(state, 'settings.forceDisplayAllColumns');
    if (columns) {
      const columnMap = {};
      columns.forEach(column => {
        columnMap[column.name] = forceDisplayAllColumns ? true : column.default;
      });
      newState = {
        ...newState,
        columnVisibility: columnMap,
      };
    }

    return newState;
  },
  [actionTypes.REFRESH_DATA](state, { data }) {
    const newState = {
      ...state,
      ...data,
      isLoad: true,
    };

    delete newState.isFetch;
    delete newState.isError;
    delete newState.triggerRefresh;

    return newState;
  },
  [actionTypes.SET_UPDATE_INTERVAL](state, { payload }) {
    return {
      ...state,
      updateInterval: payload,
    };
  },
  [actionTypes.SET_EXTRA_FETCH_PARAM](state, { payload }) {
    return {
      ...state,
      extraFetchParam: {
        ...state.extraFetchParam,
        [payload.key]: payload.value,
      },
    };
  },
  [actionTypes.FAILED](state, { isError }) {
    const newState = { ...state, isError };

    delete newState.isFetch;
    delete newState.triggerRefresh;
    delete newState.triggerResetData;

    return newState;
  },
  [actionTypes.RESET](state) {
    return { ...state, triggerReset: true };
  },
  [actionTypes.RESET_STATE]() {
    return INIT_STATE;
  },
  [actionTypes.SET_RESET_DATA](state) {
    return { ...state, triggerResetData: true };
  },
  [actionTypes.CLEAR_RESET_DATA](state) {
    const newState = { ...state };
    delete newState.triggerResetData;
    return newState;
  },
  [actionTypes.RESET_DATA](state) {
    const newState = {
      ...INIT_STATE,
      sort: state.sort,
      columnVisibility: state.columnVisibility,
      updateInterval: state.updateInterval,
      extraFetchParam: state.extraFetchParam,
      autoUpdate: state.autoUpdate,
      settings: state.settings,
    };

    if (state.grid) {
      newState.grid = state.grid;
    }

    return newState;
  },
  [actionTypes.TRIGGER_SORT_CHANGE](state, { fieldName }) {
    return {
      ...state,
      triggerSortChange: true,
      sort: {
        ...state.sort,
        newFieldName: fieldName,
      },
    };
  },
  [actionTypes.SORT_CHANGE](state, { fieldName }) {
    return {
      ...state,
      triggerLoad: true,
      triggerResetData: true,
      sort: {
        ...state.sort,
        fieldName,
        type: getNewSortType(state, fieldName),
      },
    };
  },
  [actionTypes.REFRESH](state) {
    return {
      ...state,
      triggerRefresh: true,
      triggerLoad: true,
    };
  },
  [actionTypes.RELOAD](state) {
    return {
      ...state,
      triggerLoad: true,
      triggerResetData: true,
    };
  },
  [actionTypes.CHANGE_COLUMNS](state, { payload }) {
    return {
      ...state,
      columnVisibility: payload.displayColumns,
      triggerLoad: true,
      triggerResetData: true,
    };
  },
  [actionTypes.START_UPDATE](state) {
    return {
      ...state,
      autoUpdate: true,
    };
  },
  [actionTypes.STOP_UPDATE](state) {
    const newState = { ...state };
    delete newState.autoUpdate;
    return newState;
  },
  [actionTypes.DESTROY](state) {
    const saveUI = get(state, 'settings.saveUI');
    if (saveUI) {
      return { grid: state.grid };
    }

    return undefined;
  },
  [actionTypes.ITEM_UPDATE](state, { itemAction }) {
    return ItemReducer.reducer(state, itemAction);
  },
};

const reducer = (state, action) => {
  const behavior = behaviors[action.type];
  return behavior ? behavior(state, action) : state;
};

const byInfiniteList = (state = {}, action = {}) => {
  if (!action.type.startsWith('@@infinite-list')) {
    return state;
  }

  const name = action && action.meta && action.meta.name;
  if (!name) {
    return state;
  }

  const currentState = state[name] || INIT_STATE;
  const newState = reducer(currentState, action);

  return newState === currentState ? state : { ...state, [name]: newState };
};

export default byInfiniteList;
