import { CALL_API } from 'middlewares/api';
import { actions as moduleCountersActions, modules } from 'reducers/moduleCountersSlice';
import { schema } from 'normalizr';
import { getNestedValue } from './selectors';

import {
  LIST_REQUEST,
  LIST_RECIEVE,
  LIST_FAIL,
  TOGGLE_COLLAPSE,
  TOGGLE_SELECTED,
  COLLAPSE_OPEN,
  COLLAPSE_CLOSE,
  CLEAR_SELECT,
  DESTROY,
  INIT,
  UPDATE,
  TRIGGER_UPDATE,
  SET_SELECTED,
} from './actionTypes';

const dataDecorator = data => ({ items: data });

export const getList = (name, args, requestParams) => {
  const itemSchema = new schema.Entity(
    'itemSchema',
    {},
    {
      idAttribute: entity => {
        if (!entity.id) {
          return 'root';
        }

        if (args.getSlugId) {
          return args.getSlugId(entity);
        }

        return entity.id;
      },
      processStrategy: (entity, parent) => {
        if (entity && entity.name) {
          if (parent && parent.fullPath) {
            return {
              ...entity,
              parentId: parent.id,
              fullPath: `${parent.fullPath} - ${entity.name}`,
            };
          }

          return {
            ...entity,
            parentId: parent.id,
            fullPath: entity.name,
          };
        }

        return entity;
      },
    },
  );

  itemSchema.define({
    items: [itemSchema],
  });

  let callArgs;
  if (requestParams) {
    callArgs = { data: requestParams };
  }

  return {
    type: [CALL_API],
    [CALL_API]: {
      types: [LIST_REQUEST, LIST_RECIEVE, LIST_FAIL],
      provider: args && args.provider,
      endpoint: (args && args.endpoint) || '/view/support/category/list/all',
      schema: itemSchema,
      params: args && args.params,
      callArgs,
      dataDecorator: args.backendDataType ? dataDecorator : null,
      callbacks: {
        success: (response, dispatch) => {
          const unreadCount =
            response &&
            response.data &&
            response.data.summaryCounter &&
            response.data.summaryCounter.unread;
          dispatch(
            moduleCountersActions.set({
              [modules.mail]: unreadCount,
            }),
          );
        },
      },
    },
    meta: { name },
  };
};

export const toggleCollapse = (name, id) => ({ type: TOGGLE_COLLAPSE, id, meta: { name } });

export const toggleSelected = (name, id) => ({ type: TOGGLE_SELECTED, id, meta: { name } });

export const collapseOpen = name => ({ type: COLLAPSE_OPEN, meta: { name } });

export const collapseClose = name => ({ type: COLLAPSE_CLOSE, meta: { name } });

export const clearSelect = name => ({ type: CLEAR_SELECT, meta: { name } });

export const destroy = name => ({ type: DESTROY, meta: { name } });

export const init = (name, settings) => ({ type: INIT, settings, meta: { name } });

// eslint-disable-next-line no-shadow
export const update = (name, update) => ({ type: UPDATE, update, meta: { name } });

export const triggerUpdate = name => ({ type: TRIGGER_UPDATE, meta: { name } });

export const setSelected = (name, selected) => ({ type: SET_SELECTED, selected, meta: { name } });

export const nestedOnChange = (name, onChange) => (dispatch, getState) => {
  onChange(getNestedValue(name)(getState()));
};
