import React from 'react';
import { stateSaver } from 'modules/localStorage';
import get from 'lodash/get';
import { bindActionCreators } from 'redux';
import { connect } from 'react-redux';
import shallowDiffers from 'utils/shallowDiffers';
import { NAME, SELECT_MODE } from './constants';
import saveUI from './saveUI';

import {
  getList,
  toggleSelected,
  clearSelect,
  setSelected,
  toggleCollapse,
  collapseOpen,
  collapseClose,
  init,
  destroy,
  nestedOnChange,
} from './actions';

const connectorTreeView = (settings) => (WrapComponent) => {
  const mapState = (state, props) => ({
    list: get(state, [NAME, props.name, 'list']),
    collapseMap: get(state, [NAME, props.name, 'collapseMap']),
    selectedMap: get(state, [NAME, props.name, 'selectedMap']),
    triggerUpdate: get(state, [NAME, props.name, 'triggerUpdate']),
  });

  const mapDispatch = (dispatch, props) => {
    const bindName = (actionCreator) => actionCreator.bind(null, props.name);

    return {
      getList: bindActionCreators(
        bindName(getList).bind(null, {
          provider: props.provider,
          endpoint: props.endpoint,
          backendDataType: props.backendDataType,
          getSlugId: props.getSlugId,
          params: props.params,
        }),
        dispatch,
      ),
      init: bindActionCreators(bindName(init), dispatch),
      destroy: bindActionCreators(bindName(destroy), dispatch),
      setSelected: bindActionCreators(bindName(setSelected), dispatch),
      toggleCollapse: bindActionCreators(bindName(toggleCollapse), dispatch),
      collapseOpen: bindActionCreators(bindName(collapseOpen), dispatch),
      collapseClose: bindActionCreators(bindName(collapseClose), dispatch),
      clearSelect: () => {
        dispatch(clearSelect(props.name));
        if (props.onClear) {
          props.onClear();
        }
        if (props.onChange) {
          if (props.nestedValue) {
            dispatch(nestedOnChange(props.name, props.onChange));
          } else {
            props.onChange();
          }
        }
      },
      onItemClick: (id, item, isSelected) => {
        if (props.onClick) {
          props.onClick(id, item, isSelected);
        } else {
          dispatch(toggleSelected(props.name, id));
        }
        if (props.onChange) {
          if (props.nestedValue) {
            dispatch(nestedOnChange(props.name, props.onChange));
          } else {
            props.onChange(id, item);
          }
        }
      },
    };
  };

  class TreeView extends React.Component {
    constructor(props) {
      super(props);

      if (props.saveUI) {
        stateSaver.addHandler(saveUI(props.name, props.saveUI));
      }
    }

    componentWillMount() {
      this.props.init({
        mode: this.props.mode,
        apiDefaultSlug: this.props.apiDefaultSlug,
      });

      this.props.getList(this.props.requestParams);

      if (this.props.selected) {
        if (this.props.selectedType === 'map') {
          this.props.setSelected({ selectedMap: this.props.selected });
        } else {
          this.props.setSelected(this.props.selected);
        }
      }
    }

    componentWillReceiveProps(nextProps) {
      if (this.props.selectedType === 'map') {
        if (shallowDiffers(this.props.selected, nextProps.selected)) {
          this.props.setSelected({ selectedMap: nextProps.selected });
        }
      } else if (nextProps.selected && nextProps.selected !== this.props.selected) {
        this.props.setSelected(nextProps.selected);
      } else if (this.props.enableReinitialize && nextProps.selected !== this.props.selected) {
        this.props.setSelected(nextProps.selected);
      }

      if (
        (!this.props.triggerUpdate && nextProps.triggerUpdate) ||
        shallowDiffers(this.props.requestParams, nextProps.requestParams)
      ) {
        this.props.getList(nextProps.requestParams);
      }
    }

    componentWillUnmount() {
      if (this.props.destroyOnUnmount) {
        this.props.destroy();
      }
    }

    render() {
      const { init, setSelected, selected, ...other } = this.props; // eslint-disable-line

      return <WrapComponent {...other} ref={this.props.getRef} />;
    }
  }

  const connectedTreeView = connect(mapState, mapDispatch, null, { forwardRef: true })(TreeView);
  connectedTreeView.defaultProps = {
    mode: SELECT_MODE.RADIO,
    nestedValue: false,
    ...settings,
  };

  connectedTreeView.MODE = SELECT_MODE;

  return connectedTreeView;
};

connectorTreeView.MODE = SELECT_MODE;
export default connectorTreeView;
