import Immutable from 'immutable';
import isEmpty from 'lodash/isEmpty';
import get from 'lodash/get';
import forOwn from 'lodash/forOwn';
import * as a from './actionTypes';
import { ROOT_LIST_ID, SELECT_MODE, NAME } from './constants';

const INIT_LIST = {
  isLoad: false,
  isFetch: false,
  list: null,
  collapseMap: {},
  selectedMap: {},
  selectedArray: [],
  selectedArrayOptimized: [],
  settings: {
    mode: SELECT_MODE.RADIO,
  },
};

const getCollapseMap = (items, collapseMap) => {
  const newCollapseMap = {};
  forOwn(items, (item, key) => {
    newCollapseMap[key] = true;
  });
  return { ...newCollapseMap, ...collapseMap };
};

const getSelectedArrayOptimized = state => {
  const selectedArrayOptimized = [];
  // eslint-disable-next-line no-shadow
  const walker = (state, nodeId) => {
    // eslint-disable-line no-shadow
    if (state.list[nodeId]) {
      if (state.selectedMap[nodeId]) {
        if (nodeId !== 'root') {
          selectedArrayOptimized.push(nodeId);
        }
      } else if (state.list[nodeId].items) {
        // eslint-disable-next-line no-shadow
        state.list[nodeId].items.map(nodeId => walker(state, nodeId));
      }
    }
  };
  if (state.list) walker(state, 'root');
  return selectedArrayOptimized;
};

const getSelectedArray = state => {
  const selectedArray = [];
  forOwn(state.selectedMap, (value, key) => {
    if (value && key !== ROOT_LIST_ID) {
      selectedArray.push(key);
    }
  });
  return { selectedArray, selectedArrayOptimized: getSelectedArrayOptimized(state) };
};

const toggleSelectSingle = (_imState, rootId) => {
  const multipleId = Array.isArray(rootId);

  if (multipleId) {
    const result = {};
    rootId.forEach(i => {
      result[i.toString()] = true;
    });
    return _imState.merge({ selectedMap: result });
  }
  const isSelect = !_imState.getIn(['selectedMap', rootId.toString()]);
  return _imState.setIn(['selectedMap', rootId.toString()], isSelect);
};

const toggleSelectChildren = (_imState, rootId, isSelect) => {
  const items = _imState.getIn(['list', rootId.toString(), 'items']);
  let imState = _imState.setIn(['selectedMap', rootId.toString()], isSelect);
  if (items) {
    items.forEach(value => {
      imState = toggleSelectChildren(imState, value, isSelect);
    });
  }
  return imState;
};

const rootSelectUpdate = (_imState, rootId = ROOT_LIST_ID) => {
  let imState = _imState;
  let isSelect = true;
  const itemsPath = ['list', rootId.toString(), 'items'];
  if (imState.hasIn(itemsPath)) {
    const items = imState.getIn(itemsPath);
    items.forEach(value => {
      // eslint-disable-next-line no-shadow
      const { imState: _imState, isSelect: _isSelect } = rootSelectUpdate(imState, value);
      isSelect = isSelect && _isSelect;
      imState = _imState;
    });
    imState = imState.setIn(['selectedMap', rootId.toString()], isSelect);
  } else {
    isSelect = imState.getIn(['selectedMap', rootId.toString()]);
  }
  return { imState, isSelect };
};

const updateSelected = (state, id) => {
  if (state.settings && state.settings.mode === SELECT_MODE.HIERARCHY_CHECK) {
    let imState = Immutable.fromJS(state);
    imState = toggleSelectChildren(imState, id, !imState.getIn(['selectedMap', id.toString()]));
    ({ imState } = rootSelectUpdate(imState));
    let newState = imState.toJS();
    newState = { ...newState, ...getSelectedArray(newState) };
    return newState;
  }
  if (state.settings && state.settings.mode === SELECT_MODE.CHECK) {
    let imState = Immutable.fromJS(state);
    imState = toggleSelectSingle(imState, id);
    let newState = imState.toJS();
    newState = { ...newState, ...getSelectedArray(newState) };
    return newState;
  }
  if (state.settings && state.settings.mode === SELECT_MODE.RADIO) {
    const imState = Immutable.fromJS(state);
    if (id === undefined) {
      return imState
        .set('selectedMap', {})
        .remove('selected')
        .toJS();
    }

    return imState
      .set('selectedMap', Immutable.fromJS({ [id]: true }))
      .set('selectedArray', [id])
      .set('selectedArrayOptimized', [id])
      .set('selected', id)
      .toJS();
  }

  return state;
};

const setDefaultSelected = (state, defaultIds) => {
  if (defaultIds && isEmpty(state.selectedMap)) {
    if (Array.isArray(defaultIds)) {
      // TODO: default array
    }

    return updateSelected(state, defaultIds);
  }

  return state;
};

const collapseMap = (list, flag = true) => {
  const collapseMap = {}; // eslint-disable-line no-shadow
  forOwn(list, (value, key) => {
    collapseMap[key] = flag;
  });
  return collapseMap;
};

const list = (state = INIT_LIST, action) => {
  switch (action.type) {
    case a.LIST_REQUEST:
      return { ...state, isFetch: true };
    case a.LIST_FAIL:
      return { ...state, isFetch: false, triggerUpdate: undefined };
    case a.LIST_RECIEVE: {
      const items = get(action, 'response.entities.itemSchema');
      const defaultSelected =
        state.settings.apiDefaultSlug && get(items, `root.${state.settings.apiDefaultSlug}`);
      const data = get(action, 'response.data');

      return setDefaultSelected(
        {
          ...state,
          isFetch: false,
          isLoad: true,
          list: items,
          data,
          collapseMap: getCollapseMap(items, state.collapseMap),
          default: defaultSelected,
          triggerUpdate: undefined,
        },
        defaultSelected,
      );
    }
    case a.TOGGLE_COLLAPSE:
      return Immutable.fromJS(state)
        .updateIn(['collapseMap', action.id.toString()], value => !value)
        .toJS();
    case a.TOGGLE_SELECTED:
      return updateSelected(state, action.id);
    case a.COLLAPSE_CLOSE:
      return { ...state, collapseMap: collapseMap(state.list, true) };
    case a.COLLAPSE_OPEN:
      return { ...state, collapseMap: collapseMap(state.list, false) };
    case a.CLEAR_SELECT:
      return {
        ...state,
        selectedMap: {},
        selectedArray: [],
        selectedArrayOptimized: [],
        selected: undefined,
      };
    case a.INIT:
      return { ...state, settings: action.settings };
    case a.UPDATE:
      return action.update(Immutable.fromJS(state)).toJS();
    case a.TRIGGER_UPDATE:
      return Immutable.fromJS(state)
        .set('triggerUpdate', true)
        .toJS();
    case a.SET_SELECTED:
      if (typeof action.selected === 'object') {
        if (action.selected === null) {
          // eslint-disable-next-line no-shadow
          const defaultSelected = state.default;

          return setDefaultSelected(
            {
              ...state,
              selectedMap: {},
              selectedArray: [],
              selectedArrayOptimized: [],
              selected: undefined,
            },
            defaultSelected,
          );
        }

        if (Array.isArray(action.selected)) {
          return {
            ...state,
            selectedMap: action.selected.reduce((ac, id) => {
              // eslint-disable-next-line no-param-reassign
              ac[id] = true;
              return ac;
            }, {}),
            selectedArray: action.selected,
            selectedArrayOptimized: action.selected,
          };
        }

        return {
          ...state,
          selectedMap: action.selected.selectedMap,
          selectedArray: action.selected.selectedArray,
          selectedArrayOptimized: action.selected.selectedArrayOptimized,
        };
      }

      return updateSelected(state, action.selected);
    default:
      return state;
  }
};

const arrayReducer = (state = {}, action) => {
  const { meta, ...other } = action;

  if (meta && other.type.startsWith(NAME)) {
    if (other.type === a.DESTROY) {
      const newState = { ...state };
      delete newState[meta.name];
      return newState;
    }

    return { ...state, [meta.name]: list(state[meta.name], other) };
  }

  return state;
};

export default arrayReducer;
