import { JqueryXMLHttpRequest, BackendErrorProps, BackendErrorCodeError } from './types';

export class BackendError extends Error {
  public static xhrToText(xhr: JqueryXMLHttpRequest) {
    const responseJSON = xhr.responseJSON;

    if (xhr.readyState === 0) {
      return 'Internet connection error';
    }

    if (responseJSON && responseJSON.responseStatus) {
      return `При загрузке данных произошла ошибка (${xhr.statusText || ''}).\n\n${responseJSON
        .responseStatus.message || ''}`;
    }

    if (responseJSON && responseJSON.message) {
      return responseJSON.message as string;
    }

    if (xhr.response && typeof xhr.response === 'string') {
      return xhr.response;
    }

    return 'При загрузке данных произошла ошибка';
  }

  public static createFromXhr(
    xhr: JqueryXMLHttpRequest,
    additional: BackendErrorProps['additional'] = {},
  ) {
    return new BackendError({
      message: BackendError.xhrToText(xhr),
      status: xhr.status,
      statusText: xhr.statusText,
      data: xhr.responseJSON || {},
      code: BackendErrorCodeError.NULL,
      additional,
    });
  }

  public isShowed: boolean = false;

  // eslint-disable-next-line @typescript-eslint/no-explicit-any
  public readonly data?: any;
  public readonly status: number;
  public readonly statusText: string;
  public readonly code: string;
  public readonly additional: BackendErrorProps['additional'];

  constructor(props: BackendErrorProps) {
    super(props.message);
    this.name = 'BackendError';

    this.data = props.data;
    this.status = props.status;
    this.code = props.code;
    this.statusText = props.statusText;
    this.additional = props.additional || {};
  }
}
