/* global FormData, FileList */
import checkDelegation from 'modules/delegationWidget/utils/checkDelegation';
import backendActions from './backendActions';
import showError from './showError';
import showMessages from './showMessages';
import request from './request';
import { RequestOptionsWithData, RequestOptionsBase, ResponseData } from './common.types';
import { handleUnsuccessful } from './BackendError';

// eslint-disable-next-line @typescript-eslint/no-explicit-any
export const baseCallApi = <Request = any, Response = any>(
  options: RequestOptionsWithData<Request>,
): ResponseData<Response> => {
  const { url, data, isHandleUnsuccessful, showBackendErrorForUser = true } = options;

  let promise = request(options);

  if (process.env.NODE_ENV !== 'test') {
    window.yaCounterHit(url, data);

    promise = promise.then((args) => {
      backendActions.dispatchActions(args);
      showMessages(args);
      checkDelegation(args);
      return args;
    });
  }

  // convert ( code=200 && success=false ) to error
  if (isHandleUnsuccessful) {
    promise = promise.then(handleUnsuccessful);
  }

  if (showBackendErrorForUser) {
    promise = promise.catch(showError);
  }

  return promise;
};

// eslint-disable-next-line @typescript-eslint/no-explicit-any
export const get = <Response = any>(options: RequestOptionsBase) =>
  baseCallApi<undefined, Response>({
    type: 'get',
    data: undefined,
    ...options,
  });

// eslint-disable-next-line @typescript-eslint/no-explicit-any
export const post = <Request = any, Response = any>(options: RequestOptionsWithData<Request>) =>
  baseCallApi<Request, Response>({
    type: 'post',
    ...options,
  });

// eslint-disable-next-line @typescript-eslint/no-explicit-any
export const put = <Request = any, Response = any>(options: RequestOptionsWithData<Request>) =>
  baseCallApi<Request, Response>({
    type: 'put',
    ...options,
  });

// eslint-disable-next-line @typescript-eslint/no-explicit-any
export const deleteRequest = <Request = any, Response = any>(
  options: RequestOptionsWithData<Request>,
) =>
  baseCallApi<Request, Response>({
    type: 'delete',
    ...options,
  });

// eslint-disable-next-line @typescript-eslint/no-explicit-any
export const jsonApiCall = <Request = any, Response = any>(
  options: RequestOptionsWithData<Request>,
) => {
  const { data, ...other } = options;

  return baseCallApi<string, Response>({
    ...other,
    data: JSON.stringify(data),
    contentType: 'application/json; charset=utf-8',
    type: 'post',
  });
};

/** @deprecated */
export const apiCall = (url, data, type = 'get', global = true) =>
  baseCallApi({
    url,
    data,
    type,
    global,
    isHandleUnsuccessful: true,
  });

/* TODO: refactor handleUnsuccessful() : 200 and 500 codes, remove success flag */
/** @deprecated */
export const apiCall3 = (url, data, type = 'get', global = true) =>
  baseCallApi({
    url,
    data,
    type,
    global,
  });

/* Load files */
export const apiUploadFilesCall = (url, data, global = true) => {
  const formData = new FormData();
  const param = { ...data };
  delete param.files;
  delete param.objectName;
  Object.keys(param).forEach((key) => {
    const value = param[key];
    if (value !== undefined) {
      formData.append(key, value);
    }
  });

  let { files } = data;
  if (files instanceof FileList) {
    files = Array.prototype.slice.call(files);
  }

  files.forEach((file) => formData.append(file.name, file));

  return baseCallApi({
    url,
    type: 'post',
    data: formData,
    processData: false,
    contentType: false,
    isHandleUnsuccessful: true,
    global,
  });
};
