/* global reduxStore */

import Promise from 'bluebird';
import $ from 'jquery';
import { debounceWhenArgsChange } from 'utils/debounceWhenArgsChange';
import { tabGuid } from 'utils/guid';
import wid from './wid';
import getCsrfToken from './getCsrfToken';
import { getCsrfTokenFrontend } from './getCsrfTokenFrontend';
import getRequestMetrika from './getRequestMetrika';
import { processBackendError, isCsrfError } from './BackendError';

const SET_FRONTEND_VERSION_THROTTLE_DELAY = 60000;

/*
  Использую reduxStore чтобы поправил цикличиские зависимоти request -> store -> middleware with api -> request
* Нужно подумать как уйти от reduxStore здесь.
* */
const setFrontendVersion = debounceWhenArgsChange((version) => {
  reduxStore.dispatch({ type: 'SET_FRONTEND_VERSION', payload: version });
}, SET_FRONTEND_VERSION_THROTTLE_DELAY);

const getFrontendVersion = (jqXHR) => {
  const frontendVersion = jqXHR.getResponseHeader('x-frontend-version');

  if (frontendVersion) {
    setFrontendVersion(frontendVersion);
  }
};

const updatePassportCookie = (jqXHR) => {
  const blackboxNeedReset = jqXHR.getResponseHeader('x-blackbox-need-reset');

  if (blackboxNeedReset) {
    fetch('https://passport.yandex-team.ru/auth/update/');
  }
};

export const getHeaders = () => ({
  'X-Frontend-Version': global.__APP_VERSION__, // eslint-disable-line
  'X-Frontend-TabId': tabGuid,
  'X-Frontend-Pdv': getRequestMetrika(),
  'X-Csrf-Token': getCsrfToken(),
  'X-Csrf-Token-Frontend': getCsrfTokenFrontend(),
});

const request = ({ url, host, headers = {}, ...other }) => {
  const API_HOST = host || window.CRM_SPACE_API_HOST || '';
  const urlWithWid = wid(API_HOST + url);
  const loggerPayload = {
    url: urlWithWid,
  };
  const jqxhr = $.ajax({
    url: urlWithWid,
    cache: false,
    traditional: true,
    headers: {
      ...getHeaders(),
      ...headers,
    },
    ...other,
  });

  return new Promise((resolve, reject, onCancel) => {
    jqxhr.then(
      (responseData, textStatus, jqXHR) => {
        getFrontendVersion(jqXHR);
        updatePassportCookie(jqXHR);

        if (jqXHR.status === 202) {
          reject(processBackendError(jqXHR, loggerPayload));
        }

        resolve(responseData);
      },
      (jqXHR /* , textStatus, errorThrown */) => {
        getFrontendVersion(jqXHR);
        updatePassportCookie(jqXHR);

        reject(processBackendError(jqXHR, loggerPayload));
      },
    );

    onCancel(() => {
      jqxhr.abort();
    });
  });
};

const requestWithCsrfRetry = (data, csrfRetryLimit = 2) => {
  return request(data).catch((error) => {
    if (isCsrfError(error) && csrfRetryLimit) {
      return requestWithCsrfRetry(data, csrfRetryLimit - 1);
    }

    throw error;
  });
};

export default (data) => requestWithCsrfRetry(data);
