import React from 'react';
import { IntlProvider } from 'react-intl';
import { Provider } from 'react-redux';
import store from 'store';
import { convertAnyToError } from 'services/Logger';
import { FrontendErrorWatchService } from 'services/FrontendErrorWatchService';
import { DndProvider } from 'react-dnd';
import 'lego/styles/global-fix.scss';
import { Router as ReactRouterDOM } from 'react-router-dom';
import BrowserScrollOverrider from 'components/BrowserScrollOverrider';
import { MediaMatchProvider } from 'components/MediaMatch';
import { XivaProvider } from 'modules/xiva';
import { GlobalAudioProvider } from 'modules/audio';
import { CallIssueRedirectByXivaReactConnector } from 'services/CallIssueRedirectByXiva';
import { TabMutexCleaner } from 'utils/TabMutex';
import { UXLogger } from 'components/UXLogger';
import ErrorPage from 'components/ErrorPage';
import messages from 'localization/ru';
import { ProviderWindowResize } from 'utils/windowResizeListener';

import { history } from '../appHistory';
// @ts-ignore
import Router from '../routes';
import HTML5Backend from './ReactDndModifiedBackend';
import TabFocusHelper from './TabFocusHelper';
import { PersistState } from './PersistState';
import { component as Initial } from '../modules/initial';
import { avatarService } from '../../components/Avatar';
import './style.css';

interface RootState {
  hasError: boolean;
  error?: Error;
}

class Root extends React.Component<{}, RootState> {
  static getDerivedStateFromError(error) {
    // Update state so the next render will show the fallback UI.
    return { hasError: true, error };
  }

  private frontendErrorWatchService: FrontendErrorWatchService;
  private tabMutexCleaner = new TabMutexCleaner();

  private lang: string;

  constructor(props) {
    super(props);
    this.state = { hasError: false };

    this.lang = localStorage.getItem('userLanguage') || 'ru';

    avatarService.checkStatus();
  }

  componentDidMount(): void {
    this.frontendErrorWatchService = new FrontendErrorWatchService(store);
    this.tabMutexCleaner.run();
  }

  componentWillUnmount(): void {
    this.frontendErrorWatchService.destroy();
    this.tabMutexCleaner.destroy();
  }

  componentDidCatch(error, info) {
    const errorObj = convertAnyToError(error);
    if (process.env.NODE_ENV === 'production') {
      store.dispatch({
        type: 'CRASH_REPORT',
        errorObj,
        errorLevel: 'FATAL',
        alertText: `Ошибка: ${errorObj.message}`,
        alertType: 'ERROR',
      });
    } else {
      // eslint-disable-next-line no-console
      console.log(error, info);
    }
  }

  render() {
    if (this.state.hasError) {
      // You can render any custom fallback UI
      return <ErrorPage type="frontend" error={this.state.error} />;
    }

    return (
      <>
        <DndProvider backend={HTML5Backend}>
          <GlobalAudioProvider>
            <TabFocusHelper>
              <Provider store={store}>
                <ReactRouterDOM history={history}>
                  <Initial>
                    <PersistState />
                    <UXLogger>
                      <XivaProvider>
                        <>
                          <CallIssueRedirectByXivaReactConnector />
                          <BrowserScrollOverrider />
                          <MediaMatchProvider>
                            <ProviderWindowResize>
                              <IntlProvider
                                textComponent="span"
                                locale={this.lang}
                                messages={messages}
                              >
                                <Router />
                              </IntlProvider>
                            </ProviderWindowResize>
                          </MediaMatchProvider>
                        </>
                      </XivaProvider>
                    </UXLogger>
                  </Initial>
                </ReactRouterDOM>
              </Provider>
            </TabFocusHelper>
          </GlobalAudioProvider>
        </DndProvider>
      </>
    );
  }
}

export default Root;
