import Metrika from 'modules/metrika';
import app, { Environment } from 'app';
import { config } from 'services/Config';
import { connectionChecker } from 'services/ConnectionChecker';
import { performanceMetrika } from 'services/PerformanceMetrika';
import { Config as ConfigDTO } from 'types/Config';
import { getCrmConfig } from 'utils/getCrmConfig';
import DBProvider from 'services/DBProvider';
import rumProvider from 'services/Rum';
import { runAutoSyncUserStatuses } from 'services/UserStatus';
import { get } from 'api/common';
import { BackendError, BackendErrorCodeError } from 'api/BackendError';
import { loadState } from 'modules/localStorage';
import { stateModel } from 'services/DBProvider/Models/Model.config';
import isEmpty from 'lodash/isEmpty';
import { logger } from 'services/Logger';
import * as aTypes from './actionTypes';
import { UISettingsSlice } from './UiSettings/reducer';

export const setRootInfo = () => (dispatch) => {
  dispatch({ type: aTypes.INFO_REQUEST });
  let uid;

  const initLoadData = getCrmConfig();

  let getInfoPromise: Promise<ConfigDTO>;

  if (initLoadData && 'info' in initLoadData) {
    if (initLoadData.error) {
      const error = new BackendError({
        status: 500,
        statusText: '',
        code: '',
        ...initLoadData.error,
      });
      error.name = initLoadData.error.name;
      getInfoPromise = Promise.reject(error);
    } else {
      getInfoPromise = Promise.resolve(initLoadData.info);
    }
  } else {
    getInfoPromise = get<ConfigDTO>({ url: '/info' });
  }

  return getInfoPromise
    .then(async (data) => {
      uid = (data.realUser || data.user || {}).id;
      window.USER_ID = data.user.id;
      try {
        await DBProvider.init(uid);
        rumProvider.setUpdateFromInfo(data);

        const model = await stateModel;
        model?.setUserId(data.user.id);

        const { state, panel, resize } = await DBProvider.getState();
        dispatch({ type: aTypes.INIT_STORAGE, payload: !isEmpty(state) ? state : loadState() });

        // @ts-ignore
        dispatch(UISettingsSlice.actions.initSettings({ panel, resize }));
        connectionChecker.run();
      } catch (error) {
        logger.reportError(error);
        // @ts-ignore
        dispatch(UISettingsSlice.actions.initSettings({}));
      }
      return data;
    })
    .then((data) => {
      config.value = data;

      if (!config.value.testMode) {
        app.setEnv(Environment.Prod);
        Metrika.start();
      }

      if (!config.value.config.features.logMetrika) {
        performanceMetrika.stop();
      }

      runAutoSyncUserStatuses(config.value.config.intervals.userStatus);

      Metrika.setUserId(String(uid));
      Metrika.userParams({
        UserID: Number(uid),
      });
      dispatch({ type: aTypes.INFO_RECEIVE, data });

      return data;
    })
    .catch((e) => {
      if (e && e.code === BackendErrorCodeError.DelegationErrorCode) {
        window.location.reload();
      } else {
        dispatch({ type: aTypes.INFO_FAIL });
        throw e;
      }
    });
};

export const updateRootInfoIsAutoreply = (status) => ({
  type: 'UPDATE_ROOT_INFO_IS_AUTOREPLY',
  status,
});
