import React from 'react';
import { Switch, Redirect, RouteChildrenProps } from 'react-router-dom';
import { connect } from 'react-redux';
import App from 'components/App';
import MailRouter from 'pages/mail/route';
import { Page as RequestFormRouter, PATH } from 'modules/requestForm2';
import Issues from 'modules/issues';
import Chat from 'modules/chat';
import Billing from 'modules/billing';
import { IssueModulePage } from 'components/IssueModulePage';
import { getFeatureFlag } from 'entry/store/selectors';
import { ModuleName } from 'modules/issues/types';
import { PersonalDashboardRoute, PERSONAL_DASHBOARD_PATH } from 'modules/personalDashboard';
import {
  ROUTE_PATH as ISOLATION_ADMIN_ROUTE_PATH,
  IsolationAdminRoute,
} from 'components/IsolationAdminModulePage';
import { MeetingsRouter, constants as meetingsConstants } from 'modules/meetings';
import { DTSLiftRouter, constants as DTSLiftConstants } from 'modules/dtsLift';
import ReviewRouter from 'pages/review';
import MassmailRouter from 'modules/massmail/router';
import { AccountRouter, AccountsRouter } from 'pages/accounts/route';
import PhoneRouter from 'modules/phone/router.index';
import NotFoundRouter from 'pages/notFound';
import SettingsRouter from 'modules/settings';
import adminPanelModule from 'modules/adminPanel';
import HashRouterPatched from '../components/HashRouterPatched';
import { RouteWithErrorBoundary } from '../components/RouteWithErrorBoundary';

const IssueDefault = (props) => <Issues {...props} module="issues" title="Задачи" />;

const IssueTicket = connect((state) => ({
  isNewDesign: getFeatureFlag(state, 'useNewIssueDesign'),
}))((props) => {
  const title = 'Тикеты';
  const module = ModuleName.Ticket;
  if ((props as { isNewDesign?: boolean })?.isNewDesign) {
    return <IssueModulePage {...props} module={module} title={title} />;
  }
  return <Issues {...props} module={module} title={title} />;
});

const IssueOpportunities = (props) => (
  <Issues {...props} module={ModuleName.Opportunities} title="Сделки" />
);

const NEW_TICKET_REDIRECT_PATH = ['/', '/ticket', '/request'];

const caseRedirect = ({ location }: RouteChildrenProps) => (
  <Redirect
    to={{
      ...location,
      pathname: location.pathname.replace('newtickets', ModuleName.Ticket),
    }}
  />
);

const AppRouter: React.FC = ({ children }) => (
  <HashRouterPatched>
    <App>
      <Switch>
        <RouteWithErrorBoundary path={NEW_TICKET_REDIRECT_PATH} exact>
          <Redirect to={`/${ModuleName.Ticket}`} />
        </RouteWithErrorBoundary>
        <RouteWithErrorBoundary path="/newtickets/:id?" render={caseRedirect} />
        <RouteWithErrorBoundary path="/review/:id?" component={ReviewRouter} />
        <RouteWithErrorBoundary
          path="/mail/:param1?/:param2?/:param3?/:param4?"
          component={MailRouter}
        />
        <RouteWithErrorBoundary path="/account/:id?" component={AccountRouter} />
        <RouteWithErrorBoundary path="/accounts" component={AccountsRouter} />
        <RouteWithErrorBoundary path="/massmail/:id?" component={MassmailRouter} />
        <RouteWithErrorBoundary path="/phone" component={PhoneRouter} />
        <RouteWithErrorBoundary path={PATH} component={RequestFormRouter} />
        <RouteWithErrorBoundary path="/settings" component={SettingsRouter} />
        <RouteWithErrorBoundary path="/issues" component={IssueDefault} />
        <RouteWithErrorBoundary path={`/${ModuleName.Ticket}`} component={IssueTicket} />
        <RouteWithErrorBoundary
          path={`/${ModuleName.Opportunities}`}
          component={IssueOpportunities}
        />
        <RouteWithErrorBoundary path="/chat" component={Chat} />
        <RouteWithErrorBoundary path="/billing" component={Billing} />
        <RouteWithErrorBoundary path={ISOLATION_ADMIN_ROUTE_PATH} component={IsolationAdminRoute} />
        <RouteWithErrorBoundary path={PERSONAL_DASHBOARD_PATH} component={PersonalDashboardRoute} />
        <RouteWithErrorBoundary path={adminPanelModule.url} component={adminPanelModule.router} />
        <RouteWithErrorBoundary path={meetingsConstants.ROUTE_PATH} component={MeetingsRouter} />
        <RouteWithErrorBoundary path={DTSLiftConstants.ROUTE_PATH} component={DTSLiftRouter} />
        {children}
        <RouteWithErrorBoundary path="*" component={NotFoundRouter} />
      </Switch>
    </App>
  </HashRouterPatched>
);

export default AppRouter;
